/****************************************************************************
REMODL.C

These functions were taken from the NEIC ftp site in FORTRAN and converted
to C at WC&ATWC.  

They are part of the package of programs which compute seismic travel times
and travel time tables given a modern model of the earth's spherically 
symetric shells. Many phases can be computed given the model.

REMODL generates the P/S velocity model used in computing phase times.

        Contributors:       NEIC (Buland)/ Converted to C by Whitmore
        Conversion Date:    March, 1999
        OS:                 Windows NT v4.0
        Compiler:           Microsoft Visual C++ v6.0
        Link Info:          link with iasplib.obj
        Compile Info:       See remodl.mak
---------------------------------------------------------------------------*/
#include <windows.h>
#include <winuser.h>
#include <stdlib.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include "limits.h"
#include "iasplib.h"
typedef BOOL bool;

#define PHASE_LENGTH	8	// Max String length for phase names
#define MXTMP		200

	// Function declarations
void	brkpts (int, int, FILE *);
void	crtpt (int, int, FILE *);
void	efe4 (int, int []);
void	efe8 (int, double []);
void	efec (int, char [NBR1][PHASE_LENGTH]);
void	findcp (int, FILE *);
double	findep (double, int, int, FILE *);
void	finrng (double, int, int, double, double, double, double, double, 
                int *, FILE *);
void	pgrid (int, int *, double, FILE *);
void	phcod (int *, int, double, int);
void	rough (double, int *, int *, PSZ);
double	xmod (double, int, int, int *, double *, FILE *);
void	zgrid (int, int *, FILE *);

	// Global Variables
double	dDelX[2] = {200., 200.};
double	dPMax[2] = {.01, .01};
double	dRMax[2] = {75., 75.};
double	dPb[NSL1];
double	dPm[NDP1][2];
double	dPn;
double	dPP[NSL1][2];
double	dR[NMD0];
double	dRic;
double	dRoc;
double	dRR[NSL1][2];
double	dTn;
double	dU[NMD0][2];
double	dUcrt[NCP0];
double	dVp[NMD0];
double	dVs[NMD0];
double	dX[20];
double	dXn;
double	dXX[NSL1][2];
double	dY[20];
double	dZ[NMD0];
double	dZm[NDP1][2];
double	dZm0[NDP1][2];
int	iLcb[2];
int	iLbb[2];
int	iLbrk[2][NBR2];
int	iNc = 0;
int	iNk;			// based on 1
int	iNDex[NDP1][2];
int	m;			// based on 0
int	n;			// based on 1
char	szCode[2][NBR1][PHASE_LENGTH];	// Phase identifier

int main (void)
{
static  double	dA0;
static  double	dDMax = 800.;		
static  double	dDtoL = 1.e-6;
static  double	dPmi;
static  double	dPmj;
static  double	dRn;
static  double	dTauL[2][NLVZ0];
static  double	dTauP[NSL1][2];
static  double	dToL = 1e-6;
static  double	dTTau;
static  double	dTx;
static  double	dXl[2][NLVZ0];
static  double	dXp[NSL1][2];
static  double	dXtoL;
static  double	dZic;
static  double	dZlim;
static  double	dZmax;
static  double	dZmi;
static  double	dZmj;
static  double	dZoc;
static  FILE	*hFile1Lis, *hFile2Lis, *hFileTbl, *hFileHed;	// List file handles
static  int	i, j, k, j1, k1, ii, l, nph;			// Indices based on 0
static  int	lz, n1, n2, mm;				// Counters based on 1
static  int	ndasr, nrec;
static  int	iKB[2];
static  int	iLT[2];
static  int	iLVZ[2][NLVZ0];
static  int	iMT[2];			// based on 1
static  int	iNcr;			// based on 1
static  double	plim;
static  char	szModelName[20];	// P/S velocity model name
  
rough (50., &iNk, &iNcr, szModelName);
dA0 = dR[0];
hFile1Lis = fopen ("remodl1.lis", "w");
fprintf (hFile1Lis, "\n");
fprintf (hFile1Lis, " %5ld%5ld%10.2lf%10.2lf%10.2lf\n", iNk, iNcr, dRic, dRoc, 
         dA0);
for (i=0; i<iNk; i++) fprintf (hFile1Lis, "%5ld%10.2lf%10.4lf%10.4lf\n", 
                               i+1, dR[i], dVp[i], dVs[i]);
fprintf (hFile1Lis, "\n");
dPn = dVs[0];
dXn = 1. / dR[0];
dTn = dPn * dXn;
for (i=0; i<iNk; i++)
    {
    dRn = dR[i] * dXn;
    dZ[i] = log (dRn);
    dU[i][0] = dRn * dPn / dVp[i];
    if (i+1 <= iNcr) dU[i][1] = dRn * dPn / dVs[i];
    if (i+1 > iNcr) dU[i][1] = dU[i][0];
    }
for (i=0; i<iNk; i++) 
    {
    fprintf (hFile1Lis, " %5ld%10.2lf%10.2lf", i+1, dR[i], dA0*dZ[i]);
    for (j=0; j<2; j++) fprintf (hFile1Lis, "%12.6lf", dU[i][j]);
    fprintf (hFile1Lis, "\n");
    }
fprintf (hFile1Lis, "\n");
fprintf (hFile1Lis, " delx =%10.2lf%10.2lf\n", dDelX[0], dDelX[1]);
fprintf (hFile1Lis, " dpmax -%12.6lf%12.6lf\n", dPMax[0], dPMax[1]);
fprintf (hFile1Lis, " drmax =%10.2lf%10.2lf\n", dRMax[0], dRMax[1]);

dDelX[0] = dXn * dDelX[0];
dDelX[1] = dXn * dDelX[1];
dXtoL = 1. / dA0;

findcp (iNcr, hFile1Lis);
fprintf (hFile1Lis, "\n");
fprintf (hFile1Lis, " critical points\n");
for (i=0; i<iNc; i++) fprintf (hFile1Lis, " %5ld%12.6lf\n", i+1, dUcrt[i]);

pgrid (0, &n1, dXtoL, hFile1Lis);
pgrid (1, &n2, dXtoL, hFile1Lis);
fclose (hFile1Lis);

n = 0;
j1 = 0;
k1 = 0;
for (i=0; i<iNc; i++)
    {
    ii = i+1;
    if (dPP[0][0] < dUcrt[i]-dToL)
        {
        j = 0;
        goto SkipIt;
	}
    for (j=j1; j<n1; j++)
        if (fabs (dPP[j][0]-dUcrt[ii]) <= dToL) goto SkipIt;
    j = n1 - 1;
SkipIt:
    for (k=k1; k<n2; k++)
        if (fabs (dPP[k][1]-dUcrt[ii]) <= dToL) goto NextIf;
    k = n2 - 1;
NextIf:    
    if (j-j1 <= k-k1) goto NextFor;
    for (l=j1; l<=j; l++)
        {
        n++;
        dPb[n-1] = dPP[l][0];
        }
    goto EndFor;
NextFor:    
    for (l=k1; l<=k; l++)
        {
        n++;
        dPb[n-1] = dPP[l][1];
        }
EndFor:
    j1 = j+1;
    k1 = k+1;
    }
hFile2Lis = fopen ("remodl2.lis", "w");
fprintf (hFile2Lis, "\n");
fprintf (hFile2Lis, "     1%12.6lf\n", dPb[0]);
for (i=1; i<n; i++)
    fprintf (hFile2Lis, " %5ld%12.6lf%12.2lE\n", i+1, dPb[i],(dPb[i-1]-dPb[i]));

efe8 (n, dPb);
n1 = n;
n -= 1;

zgrid (0, &iMT[0], hFile2Lis);
mm = iMT[0];
fprintf (hFile2Lis, "\n");
for (i=0; i<mm; i++)
    fprintf (hFile2Lis, " %5ld%12.6lf%12.6lf%10.2lf%12.4lE%5ld\n",
             i+1, dPm[i][0], dZm[i][0], dA0*(dZm0[i][0]-dZm0[i+1][0]),
             dA0*(dZm[i][0]-dZm[i+1][0]), iNDex[i][0]);
fprintf (hFile2Lis, " %5ld%12.6lf %12.6lf                   %5ld\n",
         mm+1, dPm[mm][0], dZm[mm][0], iNDex[mm][0]);
zgrid (1, &iMT[1], hFile2Lis);
mm = iMT[1];
fprintf (hFile2Lis, "\n");
for (i=0; i<mm; i++)
    fprintf (hFile2Lis, " %5ld%12.6lf%12.6lf%10.2lf%12.4lE%5ld\n",
             i+1, dPm[i][1], dZm[i][1], dA0*(dZm0[i][1]-dZm0[i+1][1]),
             dA0*(dZm[i][1]-dZm[i+1][1]), iNDex[i][1]);
fprintf (hFile2Lis, " %5ld%12.6lf %12.6lf                   %5ld\n",
         mm+1, dPm[mm][1], dZm[mm][1], iNDex[mm][1]);

	// Set up break pointers.
brkpts (iMT[0], 0, hFile2Lis);
fprintf (hFile2Lis, "\n ");
for (i=0; i<iLbb[0]; i++)
    fprintf (hFile2Lis, "%5ld", iLbrk[0][i]);
fprintf (hFile2Lis, "\n\n");
for (i=0; i<iLcb[0]; i++)
    fprintf (hFile2Lis, " %5ld  %s\n", i+1, szCode[0][i]);
fprintf (hFile2Lis, "\n ");
iKB[0] = iLbrk[0][iLbb[0]-1];
brkpts (iMT[1], 1, hFile2Lis);
for (i=0; i<iLbb[1]; i++)
    fprintf (hFile2Lis, "%5ld", iLbrk[1][i]);
fprintf (hFile2Lis, "\n\n");
for (i=0; i<iLcb[1]; i++)
    fprintf (hFile2Lis, " %5ld  %s\n", i+1, szCode[1][i]);
fprintf (hFile2Lis, "\n");
iKB[1] = iLbrk[1][iLbb[1]-1];
fprintf (hFile2Lis, " n1 kb %ld %ld %ld\n", n1, iKB[0], iKB[1]);

ndasr = 8 * (1+2*iKB[1]) + 4;
hFileTbl = fopen ("remodl.tbl", "wb");
fprintf (hFile2Lis, " reclength for dasign: %ld\n", ndasr);
n1 = iKB[1];
dZmax = log ((dA0-dDMax)*dXn);
dZic = log (dRic*dXn);
dZoc = log (dRoc*dXn);
dZlim = dZmax;
fprintf (hFile2Lis, "\n");
fprintf (hFile2Lis, " zmax zoc zic %19.15lE %19.15lE %19.15lE\n", 
         dZmax, dZoc, dZic);
fprintf (hFile2Lis, "\n");

	// Loop over phases.
nrec = 0;
for (nph=0; nph<2; nph++)
    {
    j = 0;
    lz = 0;
    n1 = iKB[nph];
    for (i=0; i<n1; i++)
        {
        dTauP[i][nph] = 0.;
        dXp[i][nph] = 0.;
        }
    dTauP[n1-1][nph] = dTn*1.e-6;
    dXp[n1-1][nph] = dXn*1.e-6;
    n = n1 - 1;
    mm = iMT[nph]+1;
    iNDex[0][nph] = -1;
    dZmi = dZm[0][nph];
    dPmi = dPm[0][nph];

	// Loop over model slownesses.
    for (i=1; i<mm; i++)
        {
        dZmj = dZmi;
        dZmi = dZm[i][nph];
        dPmj = dPmi;
        dPmi = dPm[i][nph];
        if (fabs (dZmj-dZmi) > 0.)
            {
		// Collect the tau and x integrals.
            for (k=0; k<n; k++)
	        {
                if (dPmi < dPb[k])
	            {
		    n = k;
		    break;
		    }
                tauint (dPb[k], dPmj, dPmi, dZmj, dZmi, &dTTau, &dTx, hFile2Lis);
//fprintf (hFile2Lis, " %ld %ld  %lE %lE %le %le %le %le %le\n", i, k, dPb[k],
 //        dPmj, dPmi, dZmj, dZmi, dTTau, dTx);
                dTauP[k][nph] = dTauP[k][nph] + dTTau;
                dXp[k][nph] = dXp[k][nph] + dTx;
                }
            if (n > 1)
                if(dPb[n-1] == dPb[n-2]) n--;
            if (dZmj < dZlim) continue;
            j++;
            if (dZmj >= dZmax) nrec++;
            dZm[j][nph] = dZmi;
            dPm[j][nph] = dPmi;
            iNDex[j][nph] = nrec;
            if (dZmj < dZmax) continue;
            fprintf (hFile2Lis, " lev1 %ld %ld %ld %lE %lE\n", j+1, n, nph+1, 
                     dPm[j][nph], dZm[j][nph]);
	    fwrite (&dZmi, sizeof (double), 1, hFileTbl);
	    fwrite (&n, sizeof (int), 1, hFileTbl);
	    for (k=0; k<n; k++) fwrite (&dTauP[k][nph], sizeof (double),
	                                1, hFileTbl);
	    for (k=0; k<n; k++) fwrite (&dXp[k][nph], sizeof (double),
	                                1, hFileTbl);
//            fprintf (hFileTbl, "%lf %ld ", dZmi, n);
//	    for (k=0; k<n; k++) fprintf (hFileTbl, "%lf ", dTauP[k][nph]);
//	    for (k=0; k<n; k++) fprintf (hFileTbl, "%lf ", dXp[k][nph]);
//            fprintf (hFileTbl, "\n");
            continue;
	    }
	  else
	    {
            if (fabs (dZmi-dZoc) <= dDtoL || fabs (dZmi-dZic) <= dDtoL)
	        {
                if (fabs (dZmi-dZm[j][nph]) <= dDtoL) continue;
                j++;
                nrec++;
                dZm[j][nph] = dZmi;
                dPm[j][nph] = dPmi;
                iNDex[j][nph] = nrec;
                fprintf (hFile2Lis, " lev2 %ld %ld %ld %lE %lE\n", j+1, n, 
                         nph+1, dPm[j][nph], dZm[j][nph]);
   	        fwrite (&dZmi, sizeof (double), 1, hFileTbl);
	        fwrite (&n1, sizeof (int), 1, hFileTbl);
	        for (k=0; k<n1; k++) fwrite (&dTauP[k][nph], sizeof (double),
	                                1, hFileTbl);
	        for (k=0; k<n1; k++) fwrite (&dXp[k][nph], sizeof (double),
	                                1, hFileTbl);
//                fprintf (hFileTbl, "%lf %ld ", dZmi, n1);
//	        for (k=0; k<n1; k++) fprintf (hFileTbl, "%lf ", dTauP[k][nph]);
//	        for (k=0; k<n1; k++) fprintf (hFileTbl, "%lf ", dXp[k][nph]);
//                fprintf (hFileTbl, "\n");
                }
	      else
	        {
                if (dZmi >= dZmax)
		    {
                    if (fabs (dZmi-dZm[j-1][nph]) > dDtoL) j++;
                    dZm[j][nph] = dZmi;
                    dPm[j][nph] = dPmi;
                    iNDex[j][nph] = iNDex[j-1][nph];
                    }
                }
            if (dPmi <= dPmj) continue;
            if (lz > 0)
                if (iLVZ[nph][lz-1] == n) continue;
            lz++;
            iLVZ[nph][lz-1] = n;
            dTauL[nph][lz-1] = dTauP[n-1][nph];
            dXl[nph][lz-1] = dXp[n-1][nph];
            fprintf (hFile2Lis, " lvz  %ld %ld %ld %lE %lE\n", lz, n, nph+1, 
                     dTauL[nph][lz-1], dXl[nph][lz-1]);
            }
        }
    j++;
    nrec++;
    dZm[j][nph] = dZmi;
    dPm[j][nph] = dPmi;
    iNDex[j][nph] = nrec;
    fprintf (hFile2Lis, " lev3 %ld %ld %ld %lE %lE\n", j+1, n, nph+1, 
                         dPm[j][nph], dZm[j][nph]);
    fprintf (hFile2Lis, "\n");
    fwrite (&dZmi, sizeof (double), 1, hFileTbl);
    fwrite (&n1, sizeof (int), 1, hFileTbl);
    for (k=0; k<n1; k++) fwrite (&dTauP[k][nph], sizeof (double),
                                1, hFileTbl);
    for (k=0; k<n1; k++) fwrite (&dXp[k][nph], sizeof (double),
                                1, hFileTbl);
//    fprintf (hFileTbl, "%lf %ld ", dZmi, n1);
//    for (k=0; k<n1; k++) fprintf (hFileTbl, "%lf ", dTauP[k][nph]);
//    for (k=0; k<n1; k++) fprintf (hFileTbl, "%lf ", dXp[k][nph]);
//    fprintf (hFileTbl, "\n");
    iMT[nph] = j + 1;
    iLT[nph] = lz;
    if (lz > 1)
        {
        efe4 (lz, &iLVZ[nph][0]);
        efe8 (lz, &dTauL[nph][0]);
        efe8 (lz, &dXl[nph][0]);
        }

    if (nph >= 1) break;
    for (i=0; i<mm; i++)
        if (dZm[i][0] < dZmax) goto Skip1;
    i = mm - 1;
Skip1:
    plim = dPm[i][0];
    fprintf (hFile2Lis, " i zmax plim zm(i, 1) = %ld %19.15lE %19.15lE %19.15lE\n",
             i+1, dZmax, plim, dZm[i][0]);
    for (i=0; i<mm; i++)
        if (dPm[i][1] <= plim) goto Skip2;
    i = mm - 1;
Skip2:
    dZlim = dZm[i][1];
    fprintf (hFile2Lis, " i plim zlim = %ld %lE %lE\n", i+1, plim, dZlim);
    }
fclose (hFileTbl);
fclose (hFile2Lis);
hFileHed = fopen ("remodl.hed", "wb");
fwrite (&ndasr, sizeof (int), 1, hFileHed);
fwrite (szModelName, 20, 1, hFileHed);
fwrite (&dZmax, sizeof (double), 1, hFileHed);
fwrite (&dZoc, sizeof (double), 1, hFileHed);
fwrite (&dZic, sizeof (double), 1, hFileHed);
fwrite (&iKB[0], sizeof (int), 1, hFileHed);
fwrite (&iKB[1], sizeof (int), 1, hFileHed);
for (i=0; i<n1; i++) fwrite (&dPb[i], sizeof (double), 1, hFileHed);
fwrite (&iMT[0], sizeof (int), 1, hFileHed);
fwrite (&iMT[1], sizeof (int), 1, hFileHed);
fwrite (&iLT[0], sizeof (int), 1, hFileHed);
fwrite (&iLT[1], sizeof (int), 1, hFileHed);
fwrite (&iLbb[0], sizeof (int), 1, hFileHed);
fwrite (&iLbb[1], sizeof (int), 1, hFileHed);
fwrite (&iLcb[0], sizeof (int), 1, hFileHed);
fwrite (&iLcb[1], sizeof (int), 1, hFileHed);
fwrite (&dXn, sizeof (double), 1, hFileHed);
fwrite (&dPn, sizeof (double), 1, hFileHed);
fwrite (&dTn, sizeof (double), 1, hFileHed);
for (nph=0; nph<2; nph++)
    {
    for (i=0; i<iLbb[nph]; i++) 
        fwrite (&iLbrk[nph][i], sizeof (int), 1, hFileHed);
    for (i=0; i<iLcb[nph]; i++)
        fwrite (szCode[nph][i], PHASE_LENGTH, 1, hFileHed);
    for (i=0; i<iMT[nph]; i++)
	    {
        fwrite (&dZm[i][nph], sizeof (double), 1, hFileHed);
        fwrite (&dPm[i][nph], sizeof (double), 1, hFileHed);
        fwrite (&iNDex[i][nph], sizeof (int), 1, hFileHed);
		}
    for (i=0; i<iLT[nph]; i++)
	    {
        fwrite (&iLVZ[nph][i], sizeof (int), 1, hFileHed);
        fwrite (&dTauL[nph][i], sizeof (double), 1, hFileHed);
        fwrite (&dXl[nph][i], sizeof (double), 1, hFileHed);
		}
    }
fclose (hFileHed);
return (0);
}

/*
 $$$$$ calls emdld and emdlv $$$$$

   Rough provides a rough interpolation of the earth model available
   through routine emdlv.  The model radii, compressional velocity, and
   shear velocity are provided in arrays r, dVp, and dVs respectively.
   Between first order discontinuities available through routine emdld,
   the radii are equally spaced as close to spacing dDr as possible.  The
   number of radii used is returned in variable n and the index of the
   core-mantle radius is returned in iNcr.  Note that emdld returns the
   radii of the discontinuities from the center of the earth out while
   rough returns the model from the surface in.  Also note that in the
   model returned by rough, each discontinuity will be represented by
   two model ajacent model points with the same radius.
*/
void rough (double dDr, int *nn, int *iNcr, PSZ pszModelName)
{
bool	bFluid;
double	dRocn = 3482., dRicn = 1271.;
int	i, j, l, np;	// based on 1 (also nn and iNcr)
double	r0, r1, dx;
double	rd[30];
double	dTol = 1.e-6;
double	dvTol = 2.e-5;

	// Get the radii of model discontinuities.
emdld (&np, rd, pszModelName);

	// Save the radii of the inner core-outer core and core-mantle boundaries
	// respectively.
dRic = 1.e6;
dRoc = 1.e6;
bFluid = FALSE;

	// Begin the interpolation.
*nn = 0;
i = np - 1;
r1 = rd[np-1];
	// Loop over each layer (between two discontinuities).
Start:
r0 = r1;
r1 = .001;
if (i > 0) r1 = rd[i-1];
if (fabs (r1-dRocn) < fabs (dRoc-dRocn)) dRoc = r1;
if (fabs (r1-dRicn) < fabs (dRic-dRicn)) dRic = r1;
if (r1 < dRoc) bFluid = TRUE;
l = (int) ((r0-r1) / dDr - .5);
dx = (r0-r1) / (l+1);
	// Set the outer most point of the layer.
*nn = *nn + 1;
dR[*nn-1] = r0;
emdlv (r0*(1.-dTol), &dVp[*nn-1], &dVs[*nn-1]);
	// Check for continuity across an apparant discontinuity.
if (*nn > 1)
    {
	// If dVp is close to continuous, force it.
    if (fabs (dVp[*nn-2]-dVp[*nn-1]) <= dvTol*dVp[*nn-1])
        {
        dVp[*nn-2] = 0.5 * (dVp[*nn-2]+dVp[*nn-1]);
        dVp[*nn-1] = dVp[*nn-2];
        }
	// If dVs is close to continuous, force it.
    if (fabs (dVs[*nn-2]-dVs[*nn-1]) <= dvTol*dVs[*nn-1])
        {
        dVs[*nn-2] = .5 * (dVs[*nn-2]+dVs[*nn-1]);
        dVs[*nn-1] = dVs[*nn-2];
	    }
    }
	// Make P and S velocity the same if we are in a fluid.
if (bFluid) dVs[*nn-1] = dVp[*nn-1];
	// Interpolate the model throughout the layer.
if (l > 0)
    for (j=1; j<=l; j++)
        {
        *nn = *nn + 1;
        dR[*nn-1] = r0 - (j)*dx;
        emdlv (dR[*nn-1], &dVp[*nn-1], &dVs[*nn-1]);
	// Make P and S velocity the same if we are in a fluid.
        if (bFluid) dVs[*nn-1] = dVp[*nn-1];
        }
	// Set the inner most point of the layer.
*nn = *nn + 1;
dR[*nn-1] = r1;
emdlv (r1*(1.+dTol), &dVp[*nn-1], &dVs[*nn-1]);
	// Make P and S velocity the same if we are in a fluid.
if (bFluid) dVs[*nn-1] = dVp[*nn-1];
	// Set the index to the core-mantle radius.
if (dRoc == r1) *iNcr = *nn;
i--;
	// If there is another layer, go do it.
if (i >= 0) goto Start;
return;
}

/*
c $$$$$ calls crtpt $$$$$
c
c        find critical points
*/
void findcp (int iNcr, FILE *hFile)
{
static	double	dTol = 1.e-6;
static	int	ifl, kfl;	// based on 1 (also iNcr)
static	int	j, i;		// based on 0

ifl = 1;
kfl = 1;
j = 0;
crtpt (0, 0, hFile);
crtpt (0, 1, hFile);
for (i=1; i<iNk; i++)
    {
    if (fabs (dZ[j]-dZ[i]) <= dTol)
        {
        crtpt (j, 0, hFile);
        crtpt (i, 0, hFile);
        if (j <= iNcr-1) crtpt (j, 1, hFile);
        if (i <= iNcr-1) crtpt (i, 1, hFile);
        goto EndOfLoop;
        }
    if (ifl == 1)
        {
        if (dU[i][0] > dU[j][0])
	    {
            ifl = 2;
            crtpt (j, 0, hFile);
	    }
	}
      else if (ifl == 2)
        {
        if (dU[i][0] < dU[j][0])
	    {
            ifl = 1;
            crtpt (j, 0, hFile);
	    }
	}
    if (i > iNcr-1) goto EndOfLoop;
    if (kfl == 1)
        {
        if (dU[i][1] > dU[j][1])
	    {
            kfl = 2;
            crtpt (j, 1, hFile);
	    }
	}
      else if (kfl == 2)
        {
        if (dU[i][1] < dU[j][1])
	    {
            kfl = 1;
            crtpt (j, 1, hFile);
	    }
	}
EndOfLoop:
    j = i;
    }
dUcrt[iNc] = 0.;
return;
}

/*
 $$$$$ calls no other routine $$$$$

   For each critical point (slowness corresponding to a first order
   discontinuity) save slowness dU[k-1][nph-1] in array dUcrt and sort it
   into descending order.  Note that k indexes an equivalent depth and
   nph indexes either P or V slownesses.
*/
void crtpt (int k, int nph, FILE *hFile)
{
static	double	dTol = 1.e-6;
static	int	i;		// based on 0 (nph, k also)
static	int	j;		// based on 1
static	double	utmp;

	// Eliminate duplicates.
if (iNc > 0)
    {
    for (i=0; i<iNc; i++)
        if (fabs (dUcrt[i]-dU[k][nph]) <= dTol)
	    {
	    fprintf (hFile, " duplicate critical value eliminated: %lE\n",
	             dU[k][nph]);
            return;
	    }
    }

iN\;
dUcrt[iNc-1] = dU[k][nph];
if (iNc <= 1) return;
j = iNc;
for (i=1; i<iNc; i++)
    {
    if (dUcrt[j-2] >= dUcrt[j-1]) return;
    utmp = dUcrt[j-2];
    dUcrt[j-2] = dUcrt[j-1];
    dUcrt[j-1] = utmp;
    j--;
    }
return;
}

void pgrid (int nph, int *n0, double xtol, FILE *hFile)
{ 
static  int	i, ic, j, k;		// Indices based on 0 (also nph)
static  int	nn, l, lsav, ll, m1, mm, kk, k1, nr; // based on 1 (also n0)
static  int	ifl;
static  double	r0, r1, r2, x0, du, sgn, u0, rsav, rnew, dA0, dx;
static  double	ps[MXTMP], xs[MXTMP], pb[3], xb[3];
static  double	dTol = 1.e-5;

dA0 = 1. / dXn;
for (i=0; i<iNc; i++)
    if (fabs (dU[0][nph]-dUcrt[i]) <= dTol) break;
ic = i;

nn = 0;
j = ic + 1;
for (i=ic; i<iNc; i++)
    {
    fprintf (hFile, "          i%5ld %12.6lf %12.6lf\n", i+1,dUcrt[i],dUcrt[j]);
    ifl = 1;
    ps[0] = dUcrt[i];
    xs[0] = xmod (dUcrt[i], nph, 1, &nr, &r0, hFile);	
    x0 = xmod (dUcrt[j], nph, -1, &nr, &r1, hFile);
    l = max ((int) (fabs (x0-xs[0]) / dDelX[nph]+0.8), 1);
    du = (ps[0]-dUcrt[j]) / (double) (l*l);
    l--;
    for (k=0; k<l; k++)
        {
        ps[k+1] = ps[0] - (double) (k+1)*(double) (k+1)*du;
        xs[k+1] = xmod (ps[k+1], nph, 1, &nr, &r2, hFile);
        }
    l += 2;
    ps[l-1] = dUcrt[j];
    xs[l-1] = x0;
AlmostTop:
    x0 = xs[1] - xs[0];
    for (k=2; k<l; k++)
        if ((xs[k]-xs[k-1])*x0 <= 0.) goto Caustic;
    goto Middle;
Caustic:
    ifl++;
    r2 = r1;
    fprintf (hFile, "          caustic\n");
    k--;
    kk = k-1;
    for (m1=1; m1<4; m1++)
        {
	kk++;
        pb[m1-1] = ps[kk-1];
        xb[m1-1] = xs[kk-1];
	}
    if ((0.5*(xb[0]+xb[2])-xb[1]) < 0.) sgn = -1.;
      else sgn = 1.;
ComeBack:
    u0 = .5 * (pb[0]+pb[1]);
    x0 = xmod (u0, nph, 1, &nr, &r1, hFile);
    if (sgn*(xb[1]-x0) >= 0.)
        {
        pb[2] = pb[1];
        xb[2] = xb[1];
        pb[1] = u0;
        xb[1] = x0;
        goto ComeBackIf;
	}
      else
        {
        pb[0] = u0;
        xb[0] = x0;
        u0 = .5 * (pb[1]+pb[2]);
        x0 = xmod (u0, nph, 1, &nr, &r1, hFile);
	}
    if (sgn*(xb[1]-x0) >= 0.) 
        {
        pb[0]=pb[1];
        xb[0]=xb[1];
        pb[1]=u0;
        xb[1]=x0;
	}
      else
        {
        pb[2]=u0;
        xb[2]=x0;
	}
ComeBackIf:
    if (fabs (xb[2]-xb[0]) > xtol) goto ComeBack;
    ps[k] = pb[1];
    xs[k] = xb[1];
    lsav = l;
    l = k + 1;
Middle:
    if (i == ic) 
        {
        nn++;
        dPP[nn-1][nph] = ps[0];
        dXX[nn-1][nph] = xs[0];
        dRR[nn-1][nph] = r0;
	fprintf (hFile, " first %5ld%10.6lf%10.6lf%9.2lf\n",
	         nn,dPP[nn-1][nph],dXX[nn-1][nph],dRR[nn-1][nph]);
        }
    k1 = max ((int) (fabs (xs[l-1]-xs[0])/dDelX[nph]+.8), 1);
    dx = (xs[l-1]-xs[0]) / (double) k1;
    x0 = xs[0];
    rsav = r0;
    mm = 2;
BackHere:
    x0 = x0 + dx;
    nn++;
    if (fabs (x0-xs[l-1]) <= dTol) 
        {
        dPP[nn-1][nph] = ps[l-1];
        dXX[nn-1][nph] = xs[l-1];
        dRR[nn-1][nph] = r1;
        goto IfSeries;
        }
    for (kk=mm; kk<=l; kk++)
        if ((x0-xs[kk-1])*(x0-xs[kk-2]) <= 0.) goto Setmm;
    kk = l;
Setmm:
    mm = kk;
    finrng (x0, nph, nn, ps[kk-1], ps[kk-2], xs[kk-1], xs[kk-2], xtol, &nr, 
            hFile);
    fprintf (hFile, " sol   %5ld%10.6lf%10.6lf%9.2lf%5ld%10.6lf%9.2lf%9.2lf\n",
             nn,dPP[nn-1][nph],dXX[nn-1][nph],dRR[nn-1][nph],nr,
             dPP[nn-2][nph]-dPP[nn-1][nph],dA0*(dXX[nn-1][nph]-dXX[nn-2][nph]),
             rsav-dRR[nn-1][nph]);
IfSeries:
    if (fabs (dPP[nn-1][nph]-dPP[nn-2][nph]) > dPMax[nph]) 
        {
        ll = max ((int) (fabs (ps[l-1]-dPP[nn-2][nph])/dPMax[nph]+.99), 1);
        dPP[nn-1][nph] = dPP[nn-2][nph]+(ps[l-1]-dPP[nn-2][nph]) / (double) ll;
        dXX[nn-1][nph] = xmod (dPP[nn-1][nph], nph, 1, &nr, &dRR[nn-1][nph], 
                               hFile);
        fprintf (hFile, "  dpmax%5ld%10.6lf%10.6lf%9.2lf%5ld%10.6lf%9.2lf%9.2lf\n",
             nn,dPP[nn-1][nph],dXX[nn-1][nph],dRR[nn-1][nph],nr,
             dPP[nn-2][nph]-dPP[nn-1][nph],dA0*(dXX[nn-1][nph]-dXX[nn-2][nph]),
             rsav-dRR[nn-1][nph]);
        k1 = max ((int) (fabs (xs[l-1]-dXX[nn-1][nph])/dDelX[nph]+.8), 1);
        dx = (xs[l-1]-dXX[nn-1][nph]) / (double) k1;
        x0 = dXX[nn-1][nph];
        mm = 2;
	}
    if (fabs (dRR[nn-1][nph]-rsav) > dRMax[nph]) 
        {
        rnew = rsav - dRMax[nph];
        while (rnew > dR[nr-1]) nr--;
        if (nr < iNk) du= fabs (dPP[nn-2][nph]-dU[nr-1][nph] * 
            pow ((rnew/dR[nr-1]),
            (log (dU[nr][nph] / dU[nr-1][nph]) / log(dR[nr]/dR[nr-1]))));
        if (nr >= iNk) du = fabs (dPP[nn-2][nph]-dU[iNk-1][nph]*rnew/dR[iNk-1]);
        ll = max ((int) (fabs (ps[l-1]-dPP[nn-2][nph])/du+.99), 1);
        dPP[nn-1][nph] = dPP[nn-2][nph]+(ps[l-1]-dPP[nn-2][nph]) / (double) ll;
        dXX[nn-1][nph] = xmod (dPP[nn-1][nph], nph, 1, &nr, &dRR[nn-1][nph], 
                               hFile);
        fprintf (hFile, "  drmax%5ld%10.6lf%10.6lf%9.2lf%5ld%10.6lf%9.2lf%9.2lf\n",
                 nn, dPP[nn-1][nph], dXX[nn-1][nph], dRR[nn-1][nph], nr,
                 dPP[nn-2][nph]-dPP[nn-1][nph],
                 dA0*(dXX[nn-1][nph]-dXX[nn-2][nph]), rsav-dRR[nn-1][nph]);
        k1 = max ((int) (fabs (xs[l-1]-dXX[nn-1][nph])/dDelX[nph]+.8), 1);
        dx = (xs[l-1] - dXX[nn-1][nph]) / (double) k1;
        x0 = dXX[nn-1][nph];
        mm = 2;
	}
    rsav = dRR[nn-1][nph];
    if (fabs (x0-xs[l-1]) > dTol) goto BackHere;

    fprintf (hFile, " end   %5ld%10.6lf%10.6lf%9.2lf%5ld%10.6lf%9.2lf%9.2lf\n",
             nn,dPP[nn-1][nph],dXX[nn-1][nph],dRR[nn-1][nph],nr,
             dPP[nn-2][nph]-dPP[nn-1][nph],dA0*(dXX[nn-1][nph]-dXX[nn-2][nph]),
             dRR[nn-2][nph]-dRR[nn-1][nph]);
    ifl--;
    if (ifl <= 0) goto EndOfLoop;
    k = 0;
    for (m1=l; m1<=lsav; m1++)
        {
        ps[k] = ps[m1-1];
        xs[k] = xs[m1-1];
        k++;
	}
    l = lsav - l + 1;
    r0 = r1;
    r1 = r2;
    goto AlmostTop;
EndOfLoop:
    j++;
    }
*n0 = nn;
return;
}

/*        partial tau integrals
*/
double xmod (double pk, int nph, int ipart, int *nr, double *rb, FILE *hFile)
{
static  int	i;		// based on 0 (also nph)
static  int	j;		// based on 1 (also ipart, nr)
static  double	x, p2, zb;

if (pk <= 0.)
    {
    x = -1.5707963;
    *nr = iNk;
    *rb = 0.;
    return (x*(-2.));
    }
p2 = pk*pk;
x = 0.;
j = 1;
for (i=1; i<iNk; i++)
    {
    if (pk > dU[i][nph]) 
        {
        *nr = j;
        *rb = dR[j-1] * pow ((pk/dU[j-1][nph]), 
              (log (dR[i]/dR[j-1])/log(dU[i][nph]/dU[j-1][nph])));
        zb =  log (*rb*dXn);
        if (dU[j-1][nph] > pk) 
	    {
            x = x + (zb-dZ[j-1])*acos(pk/dU[j-1][nph])/
                log (dU[j-1][nph]/pk);
            }
        return (x*(-2.));
        }
    if (dU[j-1][nph] != dU[i][nph]) x = x + (dZ[i]-dZ[j-1]) * 
        (acos (pk/dU[j-1][nph])
        - acos (pk/dU[i][nph])) / log (dU[j-1][nph]/dU[i][nph]);
    if (pk == dU[i][nph] && ipart < 0)
        {
        *nr = i + 1;
        *rb = dR[i];
        return (x*(-2.));
        }
    j = i + 1;
    }
*nr = iNk;
*rb = dR[iNk-1] * pk / dU[iNk-1][nph];
zb = log (*rb*dXn);
if (dU[iNk-1][nph] > pk) x = x + (zb-dZ[iNk-1]) * acos (pk/dU[iNk-1][nph])/
    log (dU[iNk-1][nph]/pk);
return (x*(-2.));
}

/*
 $$$$$ calls xmod $$$$$

   Function find0 returns x0 for which f(x0) = 0 where f(x) is a
   function supplied by the user (specified external in the calling
   routine).  X1 and x2 are the starting trial points.  Function
   evaluations are made until x0 is determined to a relative precision
   of eps by a process of inverse iterative interpolation using
   Aitken's method.
                                                     -rpb
*/
void finrng (double xg, int nph, int nn, double p0, double p1, double x0, 
             double x1, double xtol, int *nr, FILE *hFile)
{
static  char	msg[54];
static  int	j;		// based on 0 (also nph, m)
				// (nn and nr based on 1)
static	double 	ps0, xs0, ps1, xs1, xi, yi, r0;

m = 0;
if ((x1-xg)*(x0-xg) > 0.) 
    {
    printf ("Root not bracketed\n");
    exit (0);
    }
	// If the limits are already close together, there is no point in
	// iterating.
if (fabs (x0-x1) <= xtol)
    {
    dPP[nn-1][nph] = .5 * (p0+p1);
    dXX[nn-1][nph] = xmod (dPP[nn-1][nph], nph, 1, nr, &dRR[nn-1][nph], hFile);
    return;
    }
	// Set up iteration with first two trial points, x1 and x2.
dY[0] = x0 - xg;
if (fabs (dY[0]) <= xtol) 
    {
    dPP[nn-1][nph] = p0;
    dXX[nn-1][nph] = xmod (dPP[nn-1][nph], nph, 1, nr, &dRR[nn-1][nph], hFile);
    return;
    }
dX[0] = p0;
yi = x1 - xg;
if (fabs (yi) <= xtol) 
    {
    dPP[nn-1][nph] = p1;
    dXX[nn-1][nph] = xmod (dPP[nn-1][nph], nph, 1, nr, &dRR[nn-1][nph], hFile);
    return;
    }
if (dY[0] <= yi) 
    {
    ps0 = p0;
    xs0 = dY[0];
    ps1 = p1;
    xs1 = yi;
    }
  else
    {
    ps0 = p1;
    xs0 = yi;
    ps1 = p0;
    xs1 = dY[0];
    }
xi = (p0*yi - p1*dY[0]) / (yi-dY[0]);
	// Iterate.
for (m=1; m<20; m++)
    {
    if ((xi-ps0) * (xi-ps1) > 0.) xi = .5 * (ps0+ps1);
	// Save the current best guess of the zero.
    dY[m] = yi;
    dX[m] = xi;
	// Start iteration at the current best guess of the zero.
    yi = xmod (xi, nph, 1, nr, &r0, hFile) - xg;
	// Check for convergence.
    if (fabs (yi) <= xtol) goto OutOfLoop;
    if (yi <= 0.) 
        {
        ps0 = xi;
        xs0 = yi;
	}
      else
        {
        ps1 = xi;
        xs1 = yi;
	}
    for (j=0; j<m+1; j++)
        xi = (dX[j]*yi - xi*dY[j]) / (yi-dY[j]);
    }
sprintf (msg, "Iteration did not converge:  nn, nph, xg - %4ld%2ld%7.4lf\n",
         nn, nph, xg);
printf ("%s\n", msg);
exit (0);
	// Return the final best guess of the zero.
OutOfLoop:
dPP[nn-1][nph] = xi;
dXX[nn-1][nph] = yi + xg;
dRR[nn-1][nph] = r0;
return;
}

/*
         depth grid
*/
void zgrid (int nph, int *m0, FILE *hFile)
{
int	i, n1;			// Counters based on start at 1 (m0 also)
int	j, l, mm, m1;		// Indices based on 0 (nph also)
double	dTol = 1.e-6;

n1 = n + 1;
for (i=n1; i>=1; i--)
    if (fabs (dU[0][nph]-dPb[i-1]) <= dTol) break;
n1 = i;
fprintf (hFile, " zgrid %ld %lE %lE\n", n1, dU[0][nph], dPb[n1-1]);
l = 0;
i = n1 - 1;
j = 0;
dPm[0][nph] = dPb[n1-1];
dZm[0][nph] = dZ[0];
dZm0[0][nph] = 1.;
iNDex[0][nph] = n1;
Top:
if (dU[j+1][nph] <= dPb[i-1]) goto Bottom;
if (dU[j+1][nph] <= dU[j][nph]) 
    {
    j++;
    goto Top;
    }
i += 2;
for (;;)
    {
    if (dU[j+1][nph]-dPb[i-1] == 0.) break;
    if (dU[j+1][nph]-dPb[i-1] > 0.) goto Bottom;
    j++;
    }
j++;
Bottom:
l++;
dPm[l][nph] = dPb[i-1];
dZm[l][nph] = findep (dPb[i-1], j, nph, hFile);
dZm0[l][nph] = exp (dZm[l][nph]);
iNDex[l][nph] = i;
if (i > 2)
    {
    i--;
    goto Top;
    }
mm = l;
m1 = mm+1;
dPm[m1][nph] = dPb[0];
dZm[m1][nph] = -.1e6;
dZm0[m1][nph] = 0.;
iNDex[m1][nph] = 1;
*m0 = mm + 1;
}

/*
 $$$$$ calls emdlv $$$$$

   Function findep returns the equivalent depth for which the model
   slowness is u0.  If nph = 0, P slownesses are searched.  If nph = 1,
   S slownesses are searched.  K is taken as the index into equivalent
   depth near where the desired slowness should be found.  Function
   evaluations are made until u0 is fit to a relative precision of aep
   by a process of inverse iterative interpolation using Aitken's method.
*/
double findep (double u0, int k, int nph, FILE *hFile)
{
static	double	dAep = 1.e-6;
static	double	dVp, dVs;
static	int	j, kph;
static	double	x1, x2, yi, xi, dA0;

m = 0;
dA0 = 1. / dXn;
if (fabs (dZ[k]-dZ[k+1]) <= -dAep*dZ[k]) return (dZ[k]);
kph = nph;
x1 = exp (dZ[k+1]);
x2 = exp (dZ[k]);
if (dA0*x1 < dRoc-dAep) kph = 0;	// -dAep added for C version
	// Set up iteration with first two trial points, x1 and x2.
if (k+1 >= iNk) 
    emdlv (dA0*x1, &dVp, &dVs);
  else
    {
    if (fabs (dZ[k+1]-dZ[k+2]) > -dAep*dZ[k+1]) emdlv (dA0*x1, &dVp, &dVs);
      else emdlv (dA0*x1*(1.+dAep), &dVp, &dVs);
    }
if (kph == 0) dY[0] = dPn*x1/dVp - u0;
if (kph == 1) dY[0] = dPn*x1/dVs - u0;
if (fabs (dY[0]) <= dAep*u0) return (dZ[k+1]);
dX[0] = x1;
if (k <= 1) emdlv (dA0*x2, &dVp, &dVs);
  else
    {
    if (fabs (dZ[k-1]-dZ[k]) > -dAep*dZ[k]) emdlv (dA0*x2, &dVp, &dVs);
      else
        emdlv (dA0*x2*(1.-dAep), &dVp, &dVs);
    }
if (kph == 0) yi = dPn*x2/dVp - u0;
if (kph == 1) yi = dPn*x2/dVs - u0;
if (fabs (yi) <= dAep*u0 || yi == dY[0]) return (dZ[k]);
xi = (x1*yi - x2*dY[0]) / (yi - dY[0]);
if (fabs (xi-x1) <= dAep*max (fabs (xi), dAep) || fabs(xi-x2) <=
    dAep*max (fabs (xi), dAep)) goto NextToLast;
	// Iterate.
for (m=1; m<20; m++)
    {	// Save the current best guess of the zero.
    dY[m] = yi;
    dX[m] = xi;
	// Start iteration at the current best guess of the zero.
    emdlv (dA0*xi, &dVp, &dVs);
    if (kph == 0) yi = dPn*xi/dVp - u0;
    if (kph == 1) yi = dPn*xi/dVs - u0;
    for (j=0; j<m+1; j++)
        {
        if (yi == dY[j]) goto NextToLast;
        xi = (dX[j]*yi - xi*dY[j]) / (yi - dY[j]);
        }
	// Check for convergence.
    if (fabs (xi-dX[m]) <= dAep*max (fabs (xi), dAep)) goto NextToLast;
    }
m = -1;
	// Return the final best guess of the zero.
NextToLast:
return (log (max (xi,dAep)));
}

/*
 $$$$$ calls efec, efe4, and phcod $$$$$

        sets up discontinuity information
*/
void brkpts (int mm, int nph, FILE *hFile)
{
static	double	dTol = 1.e-6;
static	int	i, lb;			// based on 0 (nph also)
static	int	lc, isw;		// based on 1 (mm also)

lb = 0;
iLbrk[nph][0] = iNDex[0][nph];
lc = 0;
i = 0;
isw = 1;
	// Search for discontinuities.
TopOfLoop:
if (fabs(dZm[i][nph]-dZm[i+1][nph]) > (-1.)*dZm[i+1][nph]*dTol)
    {
    if (i+1 >= mm) goto End;
    i++;
	// No discontinuity.
    if (isw == 1)
	// Have we hit a high slowness zone?
        {
        if (dPm[i][nph] <= dPm[i-1][nph]) goto TopOfLoop;
	// Yes, flag it.
        isw = 2;
	// If the high slowness zone is topped with a discontinuity, go back
	// to the usual processing.
        if (iLbrk[nph][lb] == iNDex[i-1][nph]) goto TopOfLoop;
	// Otherwise, mark the grazing ray to the top of the zone.
        lb++;
        iLbrk[nph][lb] = iNDex[i-1][nph];
        phcod (&lc, nph, dZm[i-1][nph], -1);
        goto TopOfLoop;
        }
	// We are already in a high slowness zone.  See if we have hit bottom.
    if (dPm[i][nph] >= dPm[i-1][nph]) goto TopOfLoop;
	// Yes we have.  Reset the high slowness zone flag.
    isw = 1;
    goto TopOfLoop;
    }
	// Discontinuity!  See what kind.
if (dPm[i+1][nph] <= dPm[i][nph]) 
    {
	// Velocity increase.  Flag the bottom of the step.
    lb++;
    iLbrk[nph][lb] = iNDex[i][nph];
    phcod (&lc, nph, dZm[i][nph], 1);
    if (iLbrk[nph][lb] >= iLbrk[nph][lb-1])
        {
        lb--;
        lc--;
        strcpy (szCode[nph][lc-1], szCode[nph][lc]);
        }
	// Find the top of the discontinuity.
    if (i+1 >= mm) goto End;
    i++;
    while (fabs (dZm[i][nph]-dZm[i+1][nph]) <= (-1.)*dZm[i+1][nph]*dTol)
        {
        if (i+1 >= mm) goto End;
        i++;
        }
	// Flag the top of the step.
    lb++;
    iLbrk[nph][lb] = iNDex[i][nph];
    if (iLbrk[nph][lb] < iLbrk[nph][lb-1]) goto TopOfLoop;
    lb--;
    lc--;
    goto TopOfLoop;
    }
	// Velocity decrease. Flag the top of the step.
lb++;
iLbrk[nph][lb] = iNDex[i][nph];
phcod (&lc, nph, dZm[i][nph], -1);
	// Find the bottom of the discontinuity.
for (;;)
    {
    if (i+1 >= mm) goto End;
    i++;
    if ((fabs (dZm[i][nph]-dZm[i+1][nph])+dZm[i+1][nph]*dTol) > 0.)
        goto TopOfLoop;
    }
	// We have hit the bottom of the model.
End:
phcod (&lc, nph, dZm[mm-1][nph], -1);
efe4 (lb+1, &iLbrk[nph][0]);
efec (lc, &szCode[nph][0]);
iLbb[nph] = lb+1;
iLcb[nph] = lc;
return;
}

/*       set up phase codes
*/
void phcod (int *lc, int nph, double z0, int kfl)
{
static	char	szTag[2];
static	double	d0, r0;
static	int	i;		// Index based on 0
static	int	j; 		// Counter based on 1
static	int	idep;
static	char	szPre[3], szSuf[2], szBuf[10], szDep[4];

d0 = (1.-exp (z0)) / dXn;
r0 = 1. / dXn - d0;
idep = (int) (d0 + 0.5);
if (*lc <= 0) 
    {
    *lc = 1;
    strcpy (szPre, "  ");
    strcpy (szSuf, " ");
    if (nph == 0) strcpy (szTag, "P");
    if (nph == 1) strcpy (szTag, "S");
    strcpy (szCode[nph][*lc-1], "t");
    strcat (szCode[nph][*lc-1], szTag);
    strcat (szCode[nph][*lc-1], "g");
    goto End;
    }
*lc = *lc + 1;
if (idep <= 70) 
    {
    strcpy (szCode[nph][*lc-1], "t");
    strcat (szCode[nph][*lc-1], szTag);
    strcat (szCode[nph][*lc-1], "b");
    if (szCode[nph][*lc-2][2] == 'b') szCode[nph][*lc-2][2]='g';
    if (szCode[nph][*lc-3][2] == 'b') szCode[nph][*lc-3][2]='g';
    goto End;
    }
if (!strcmp (szPre, "  ")) 
    {
    strcpy (szCode[nph][*lc-1], "t");
    strcat (szCode[nph][*lc-1], szTag);
    }
  else
    {
    strcpy (szCode[nph][*lc-1], "t");
    strcat (szCode[nph][*lc-1], szTag);
    strcat (szCode[nph][*lc-1], szPre);
    strcat (szCode[nph][*lc-1], szSuf);
    strcat (szCode[nph][*lc-1], szTag);
    }
if (szCode[nph][*lc-3][2] == 'g' || szCode[nph][*lc-3][2] == 'b')
    {
    szCode[nph][*lc-1][2] = 'n';
    strcpy (szCode[nph][*lc-2], "r");
    strcat (szCode[nph][*lc-2], szTag);
    strcat (szCode[nph][*lc-2], "m");
    strcat (szCode[nph][*lc-2], szTag);
    }
j = 0;
for (i=0; i<PHASE_LENGTH; i++)
    {
    if (szCode[nph][*lc-1][i] != ' ')
        {
	j++;
        szCode[nph][*lc-1][j-1] = szCode[nph][*lc-1][i];
	}
    }
if (j < PHASE_LENGTH)
    for (i=j; i<PHASE_LENGTH; i++) szCode[nph][*lc-1][i] = ' ';
if (!strcmp (&szCode[nph][*lc-1][1], "PKP") && 
    !strcmp (&szCode[nph][*lc-2][1], "Pn"))
    for (i=2; i<PHASE_LENGTH; i++) szCode[nph][*lc-2][i] = ' ';
if (!strcmp (&szCode[nph][*lc-1][1], "SKS") && 
    !strcmp (&szCode[nph][*lc-3][1], "Sn"))
    for (i=2; i<PHASE_LENGTH; i++) szCode[nph][*lc-3][i] = ' ';
if (!strcmp (&szCode[nph][*lc-1][1], "PKP"))
    strcpy (&szCode[nph][*lc-1][1], "PKPab");
if (!strcmp (&szCode[nph][*lc-1][1], "PKIKP"))
    strcpy (&szCode[nph][*lc-1][1], "PKPdf");
if (!strcmp (&szCode[nph][*lc-1][1], "SKS"))
    strcpy (&szCode[nph][*lc-1][1], "SKSab");
if (!strcmp (&szCode[nph][*lc-1][1], "SKIKS"))
    strcpy (&szCode[nph][*lc-1][1], "SKSdf");
if (fabs (r0-dRoc) <= 20.) 
    {
    strcpy (szPre, "K");
    if (kfl <= 0) return;
    *lc = *lc + 1;
    strcpy (szCode[nph][*lc-1], "r");
    strcat (szCode[nph][*lc-1], szTag);
    strcat (szCode[nph][*lc-1], "c");
    strcat (szCode[nph][*lc-1], szTag);
    return;
    }
if (fabs (r0-dRic) <= 20.)
    {
    strcpy (szPre, "KI");
    strcpy (szSuf, "K");
    if (kfl <= 0) return;
    *lc = *lc + 1;
    strcpy (szCode[nph][*lc-1], "r");
    strcat (szCode[nph][*lc-1], szTag);
    strcat (szCode[nph][*lc-1], "KiK");
    strcat (szCode[nph][*lc-1], szTag);
    return;
    }
End:
if (kfl <= 0) return;
*lc = *lc + 1;
itoa (idep, szDep, 10);
strcpy (szBuf, szTag);
strcat (szBuf, szPre);
strcat (szBuf, "d");
strcat (szBuf, szDep);
strcat (szBuf, szSuf);
strcat (szBuf, szTag);
strcpy (szCode[nph][*lc-1], "r");
j = 1;
for (i=0; i<10; i++)
    if (szBuf[i] != ' ' && j < PHASE_LENGTH) 
        {
	j++;
	szCode[nph][*lc-1][j-1] = szBuf[i];
	}
if (j < PHASE_LENGTH) 
    for (i=j; i<PHASE_LENGTH; i++) szCode[nph][*lc-1][i] = ' ';
return;
}

/*
 $$$$$ calls no other routine $$$$$

   Integer array na(n) is transposed end-for-end.
*/
void efe4 (int nn, int na [])
{
int	j, n2, i, nb;

if (nn <= 1) return;
n2 = nn / 2;
j = nn;
for (i=0; i<n2; i++)
    {
    j--;
    nb = na[i];
    na[i] = na[j];
    na[j] = nb;
    }
}

/*
 $$$$$ calls no other routine $$$$$

   Double precision array da(n) is transposed end-for-end.
*/
void efe8 (int nn, double da [])
{
double 	db;
int	i, j, n2;

if (nn <= 1) return;
n2 = nn / 2;
j = nn;
for (i=0; i<n2; i++)
    {
    j--;
    db = da[i];
    da[i] = da[j];
    da[j] = db;
    }
return;
}

/*
 $$$$$ calls no other routine $$$$$

   Character array ia(n) is transposed end-for-end.  Ib must be a
   character variable with the same character width as each element of
   array ia.
*/
void efec (int nn, char sz1[NBR1][PHASE_LENGTH])
{
char	szTemp2[PHASE_LENGTH];
int	i, j, n2;

if (nn <= 1) return;
n2 = nn/2;
j = nn;
for (i=0; i<n2; i++)
    {
    j--;
    strcpy (szTemp2, sz1[i]);
    strcpy (sz1[i], sz1[j]);
    strcpy (sz1[j], szTemp2);
    }
}
