//      main.cpp - V0 cosmos file converter
//                      produces XML files from COSMOS V0 ascii files
//      Author: Pete Lombard, who basicly chopped out the mseed stuff from
//      v02mseed.
//
//      Based on v02mseed by Paul Friberg, ISTI

//	Change record from v02mseed:
//
//		See COPYRIGHT FILE IN THIS DIR.
//
//      version 1 	December 17, 2001 (started)
//	version 1.01 	December 31, 2001 (released)
//	version 1.02	July 22, 2003 (V0 xml options)
//	version 1.03    August 6, 2003 (V0 xml reading capability added)
//	version 1.04	August 12, 2003 (added -v for verbose + diff capability changed)
//	version 1.05	August 24, 2003 (added -d for diffing of two xml files)
//      version 1.06    March 5, 2004 (fixed diff capability to announce differences if they exist)
//	version 1.07    July 3, 2004 (added dataless checks and dataless timespan parsing and output in XML)
//	version 1.08    July 12, 2004 (bug fix version to multiplexed dataless)
//	version 1.09	July 22, 2004 (added comma sep list to -C option for diff channel names per multiplexed V0 or dataless)
//	version 1.10    July 23, 2004 ( minor changes to above change)
//	version 1.11	March 28, 2005 (minor change to station name if using number, prefix with 0's)
//      version 1.12    May 13, 2010 (Added additional values from Cosmos tables) - SMZ
//
//      Change record for v02xml:
//      version 1.00    October 24, 2011


#define VER_NO "1.00 - 2011-10-24"


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <vector>

#include "filelist.h"
#include "v0_multiplexor.h"
#include "v0_cosmos.h"
#include "v0_xml.h"

using namespace std;

char * progname;

// StringToUpper
void StringToUpper(char * ctmp) {
    for (unsigned int j=0; j< strlen(ctmp) ; j++) {
	ctmp[j] = toupper(ctmp[j]);
    }
}

// CSL_to_vector - converts a comma sep list to a vector
vector <char *> * CSL_to_vector (char *Str) {
    vector <char *> *v;
    char *last, *cptr, *tmp;
    v = new vector<char *>;

    if (Str == NULL)
	return v;  //empty vector
    // remove the quotes if any
    if ( (cptr = strstr(Str, "\"")) != NULL) {
	// found one on the front end
	if (cptr == Str)
	    Str++;
    }
    if ( (cptr = strstr(Str, "\"")) != NULL) {
	// found one on the backend...
	*cptr = '\0';
    }
    cptr = strstr(Str, ",");
    if ( cptr== NULL) {
	// there is only 1 chan!
	// copy it and then insert it
	tmp=new char[strlen(Str)+1];
	strcpy(tmp, Str);
	StringToUpper(tmp);
	v->insert(v->begin(), tmp);
    } else {
	*cptr = '\0';
	cptr++;
	tmp=new char[strlen(Str)+1];
	strcpy(tmp, Str);
	StringToUpper(tmp);
	v->insert(v->begin(), tmp);
	last = cptr;
	// more than one found, add them to the vector
	while ((cptr = strstr(last, ",")) != NULL) {
	    *cptr = '\0';
	    tmp=new char[strlen(last)+1];
	    strcpy(tmp, last);
	    StringToUpper(tmp);
	    //v->insert(v->begin(), tmp);
	    v->push_back(tmp);	// insert it at the end
	    cptr++;
	    last = cptr;
	} // end of while strtok() loop
	tmp=new char[strlen(last)+1];
	strcpy(tmp, last);
	StringToUpper(tmp);
	//v->insert(v->begin(), tmp);
	v->push_back(tmp);
    }
    return v;
}


void usage() {
    fprintf(stderr, "%s: Version %s\n", progname, VER_NO);
    fprintf(stderr, "%s: usage [-vhH][-i xmlinput][-d xml_to_diff][-x xmloutput][-C chan][-L location-code][-N network-code][-S station][-f] -F filelist | V0 file...\n", progname);
    fprintf(stderr, " -v sets verbosity on\n");
    fprintf(stderr, " -h displays usage message like this\n");
    fprintf(stderr, " -i xmlinput_filename - previous XML header file created by this program\n");
    fprintf(stderr, " -d xml_to_diff - previous XML header file created by this program to diff -i input xml data against\n");
    fprintf(stderr, " -x xmloutput_filename - outputs an XML header file for 1 v0 file\n");
    fprintf(stderr, "  -x and -i only work with one file at a time\n");
    fprintf(stderr, "  -C chan[,chan]...  if multiple channels are specified,\n    they must be comma separated, with no spaces.\n");

}                                                                                                      


int main (int argc, char ** argv) {


    v0_cosmos *v0;
    v0_xml *vxi, *v0xml, *vxid;
    char *cptr,*ofname, *nc, *sta, *chan;
    char *sta_override_name;
    char *chan_override_name;
    char *net_override_name;
    char *loc_override_name;
    char *xml_output_filename, *xml_input_filename, *xml_diff_filename;
    int ret_size;
    FileList *flist;
    char c, *file_name, *filelist_filename;
    bool verbose;
    bool xml_outfile_created;
    bool generate_xml_files;
    vector <char *> * chan_vector;


    generate_xml_files = true;	// for each v0 or dataless v0 file, generate a .xml file to go with it
    xml_outfile_created = false;	// set to true if we allocate a new filename based on the input dataless.
    verbose = false;
    vxi = NULL;			// contains the v0_XML for the input xml file
    vxid = NULL;			// contains the v0_XML for the input differencing xml file
    filelist_filename = NULL;	// a list of files to convert
    sta_override_name = NULL; 	// override the site name
    net_override_name = NULL; 	// override the network name
    chan_override_name = NULL; 	// override the channel name
    loc_override_name = NULL; 	// override the location code
    xml_output_filename = NULL;	// where to put an XML header file
    xml_input_filename = NULL;	// an XML file header for differencing against
    xml_diff_filename = NULL;	// another XML file header for differencing against
    chan_vector = NULL;

    progname = argv[0];
    if ( (cptr=strrchr(argv[0],'/')) != NULL) {
	// truncate off any absolute path
	cptr++;
	progname=cptr;
    }
	
    if (argc < 2) {
	usage();
	exit(1);
    }
	


    // process command line options (get necessary params)
    while((c = getopt(argc, argv, "?vHhd:i:S:C:L:N:F:x:")) != -1 ) {
	switch(c) {
	case 'v':
	    verbose=true;
	    break;
	case 'S':
	    sta_override_name = optarg;
	    if (strlen(sta_override_name) > 5) {
		fprintf(stderr, "SEED Station names must be 5 chars or less\n");
		exit(2);
	    }
	    break;
	case 'N':
	    net_override_name = optarg;
	    if (strlen(net_override_name) > 2) {
		fprintf(stderr, "SEED Network codes must be 2 chars or less\n");
		exit(2);
	    }
	    break;
	case 'C':
	    chan_override_name = optarg;
	    if (strchr(chan_override_name, ',') != NULL) {
		chan_vector = CSL_to_vector( chan_override_name );
	    } else {
		if (strlen(chan_override_name) > 3) {
		    fprintf(stderr, "SEED Channel codes must be 3 chars or less\n");
		    exit(2);
		}
	    }
	    break;
	case 'F':
	    filelist_filename = optarg;
	    break;
	case 'x':
	    xml_output_filename = optarg;
	    break;
	case 'd':
	    xml_diff_filename = optarg;
	    break;
	case 'i':
	    xml_input_filename = optarg;
	    break;
	case 'L':
	    loc_override_name = optarg;
	    if (strlen(loc_override_name) > 3) {
		fprintf(stderr, "SEED Location codes must be 2 chars or less\n");
		exit(2);
	    }
	    break;
	    break;
	case '?':
	case 'h':
	case 'H':
	    usage();
	    exit(0);
	default:
	    fprintf(stderr, "Ignoring invalid option -%c \n", c);
	    break;
	}
    }

    if (optind <argc) /* get file list off command line */
	flist = new FileList(optind, argc, argv);
    else {
	flist =  new FileList(filelist_filename);
    } 

    if (xml_input_filename) {
	vxi= new v0_xml(xml_input_filename);
	// vxi->printElementInfo("full_scale_input");
    }
    if (xml_diff_filename) {
	vxid= new v0_xml(xml_diff_filename);
    }

    if (vxid && vxi) {
	// simply difference and then exit with return of difference!
	exit( vxid->diff(vxi, verbose) ) ;
    }

    // loop over any files however they are provided
    while ( (file_name = flist->getNextFile()) != NULL )
    {

	v0_multiplexor *vmx = NULL;

	try {
	    vmx = new v0_multiplexor(file_name);
	    try {
		while ( (v0 = vmx->getNext()) != NULL) {
		    if (v0->getData() == NULL && ! v0->isDataless()) {
			fprintf(stderr, "Could not process %s COSMOS V0 information\n", file_name);
			continue;
		    }
		    if (v0->getStageIndex() != 0) {
			fprintf(stderr, "Could not process %s COSMOS, not a stage V0 file\n", file_name);
			continue;
		    }
	
		    // set the station name if asked to
		    if (sta_override_name != NULL) {
			v0->setStationCode(sta_override_name);
		    }

		    // set the channel name
		    if (chan_override_name != NULL) {
			if (chan_vector != NULL) {
			    int cnum;
			    cnum = v0->getChannelNumber();
			    if (cnum > chan_vector->size()) {
				fprintf(stderr, "Error: SEED Channel list too short, %d elements, but channel num %d listed in file\n",
					chan_vector->size(), cnum);
				exit(3);
			    } else {
				chan = (*chan_vector)[cnum-1];
			    }
			} else {
			    chan = chan_override_name;
			}
			// set it in the v0 too since this is not there yet
			v0->setChannelCode(chan);
		    } else  {
			chan = v0->getChannelCode();
		    }

		    // write a dataless name .xml IF the xml_output_filename is NULL
		    if (xml_output_filename==NULL && (v0->isDataless() || generate_xml_files) )  {
			xml_output_filename = new char[strlen(file_name)+8];
			sprintf(xml_output_filename, "%s.%d.xml", file_name, v0->getChannelNumber());
			xml_outfile_created = true;
		    }

		    if (xml_input_filename || xml_output_filename) {
			v0xml = new v0_xml(v0);

			if (vxi) {
			    if(v0xml->diff(vxi, verbose) == 1 && verbose==false) {
                                // do a second pass with verbose=true
                                v0xml->diff(vxi, true);
			    }

			}
		    }

		    if (xml_output_filename) {
			v0xml->outputToFile(xml_output_filename);
			delete v0xml;
			v0xml = NULL;
			if (xml_outfile_created) {
			    delete xml_output_filename;
			    xml_output_filename = NULL;
			}
		    }
		} // end of loop processing v0 files
	    } catch ( v0_cosmos::format_exception fme ) {
		fprintf(stderr, "Format Exception Caught with file %s\n", file_name); 
		fme.display();
		fprintf(stderr, "Skipping this file.\n"); 
	    }
	} catch (v0_multiplexor::file_io_exception f) {
	    fprintf(stderr, "Problem attempting to read from file %s\n", file_name);
	    fprintf(stderr, "Skipping this file.\n"); 
	}
	if (vmx != NULL) delete vmx;
    }// end of while loop

    exit(0);

}
