/* ====================================================================
 * 
 * Copyright (C) 2004 Instrumental Software Technologies, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code, or portions of this source code,
 *    must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by Instrumental
 *    Software Technologies, Inc. (http://www.isti.com)"
 *   
 * 4. If the software is provided with, or as part of a commercial
 *    product, or is used in other commercial software products the
 *    customer must be informed that "This product includes software
 *    developed by Instrumental Software Technologies, Inc. 
 *    (http://www.isti.com)"
 *   
 * 5. The names "Instrumental Software Technologies, Inc." and "ISTI"
 *    must not be used to endorse or promote products derived from
 *    this software without prior written permission. For written
 *    permission, please contact "info@isti.com".
 *
 * 6. Products derived from this software may not be called "ISTI"
 *    nor may "ISTI" appear in their names without prior written
 *    permission of Instrumental Software Technologies, Inc.
 *
 * 7. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Instrumental
 *    Software Technologies, Inc. (http://www.isti.com/)."
 *
 * THIS SOFTWARE IS PROVIDED BY INSTRUMENTAL SOFTWARE
 * TECHNOLOGIES, INC. ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL INSTRUMENTAL SOFTWARE TECHNOLOGIES,
 * INC.  OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 ====================================================================
 
  A current version of the software can be found at 
  <A HREF="www.isti.com">http://www.isti.com</A>

  Bug reports and comments should be directed to 
  Instrumental Software Technologies, Inc. at info@isti.com

 ====================================================================*/


#include <iostream.h>
#include <sys/types.h>
#include <regex.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "v0_cosmos.h"


const int MAX_LINE_SIZE = 256;	// max possible size
const int V0_LINE_SIZE = 80;		// most probable size
const int NUM_TEXT_LINES_COL = 46;
const int V0_NETWORK_CODE_COL = 25;
const int V0_STATION_CODE_COL = 28;
const int V0_STATION_DESCRIP_COL = 40;
const int V0_STATION_DESCRIP_LEN = 40;
const int V0_INTEGER_NUM_LINES_COL = 37;
const int V0_INTEGER_FORMAT_SIZE_COL = 56;
const int V0_REAL_NUM_LINES_COL = 34;
const int V0_REAL_FORMAT_SIZE_COL = 53;
const int V0_DATA_FORMAT_SIZE_COL = 70;

const int V0_INTEGER_ORIENTATION = 53;
const int V0_INTEGER_STA_NUMBER = 7;

const int V0_INTEGER_UNKNOWN_COL = 65;
const int V0_REAL_UNKNOWN_COL = 73;

char * _getNetCode(int i);

// This constructor is for parsing multiplexed data using the  clear() 
// and parse() methods. ???
v0_cosmos::v0_cosmos() throw(v0_cosmos::format_exception) {
    
    file_name = NULL;
    is_dataless = false;
    net_code = NULL;
    sta_name = NULL;
    chan_code = NULL;
    station_description[0] = '\0';
    comment_lines = NULL;
    real_format[0] = '\0';
    dataless_start = NULL;
    dataless_end = NULL;
    sta_number = 0;
    chan_num = 0;
    orientation = 0;
    number_samples = 0;
    unknown_int = -999;
    unknown_real = -999.0;
    data = NULL;
}

// This constructor is for parsing NON-multiplexed V0 files one at a time.
v0_cosmos::v0_cosmos(char * filename) throw(format_exception) {
    FILE *fptr;

    file_name = filename;
    is_dataless = false;
    net_code = NULL;
    sta_name = NULL;
    chan_code = NULL;
    station_description[0] = '\0';
    comment_lines = NULL;
    real_format[0] = '\0';
    dataless_start = NULL;
    dataless_end = NULL;
    sta_number = 0;
    chan_num = 0;
    orientation = 0;
    number_samples = 0;
    unknown_int = -999;
    unknown_real = -999.0;
    data = NULL;

    // open the sucker up
    if ((fptr=fopen(file_name, "r")) == NULL) {
	fprintf(stderr, "Could not open V0 file %s\n", filename);
	return;
    }
    if(!parse(fptr)) {
	fprintf(stderr, "Errors found in V0 while parsing file %s\n", filename);
    }
    fclose(fptr);
}



void v0_cosmos::setFilename(char * filename) {
    file_name = filename;
}

static char err_string[1024];
bool v0_cosmos::parse(FILE *fptr) throw( v0_cosmos::format_exception ) {

    char linebuf[MAX_LINE_SIZE+2], *cptr, *cptr2, *cptr3;
    int num_data_per_line,num_lines, i,j, int_ptr,fmt_size, num_per_line;
    int ret;
    char tmpsta[6];
    regex_t preg;
    regmatch_t pmatch[4];
    char regex_err[1024];
	
    // get the first line containing number of text lines to process
    if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	return false;
    }

    // get number of text lines to read
    cptr = &linebuf[NUM_TEXT_LINES_COL];
    num_lines = atoi(cptr) - 1;
    if (num_lines <= 0) {
	sprintf(err_string, "Error: First line of %s did not indicate number of text lines.", file_name);
	throw(format_exception(err_string));
    }

    // check the 2nd line if this is Dataless or not
    if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	throw(format_exception("Error: Second line of this V0 entry was empty."));
    }
    if (strncmp(linebuf, "Dataless", 8) ==0 ) is_dataless=true;

    // skip 2 lines (lines 3,4 contain equake info regarding this record)
    if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	throw(format_exception("Error: Third line of this V0 entry was empty."));
    }
    if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	throw(format_exception("Error: Fourth line of this V0 entry was empty."));
    }

    // get network code and station description (line 5)
    if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	throw(format_exception("Error: Fifth line of this V0 entry was empty."));
    }
    cptr = &linebuf[V0_NETWORK_CODE_COL];
    net_code = new char[3];
    strncpy(net_code, cptr, 2);
    net_code[2]='\0';
	
    // get station code if it exists
    cptr = &linebuf[V0_STATION_CODE_COL];
    strncpy(tmpsta, cptr, 5);
    tmpsta[5]='\0';
    if (tmpsta[0] != ' ') {
	sta_name = new char[6];
	strcpy(sta_name, tmpsta);
    }

    strncpy(station_description, &linebuf[V0_STATION_DESCRIP_COL], 
	    V0_STATION_DESCRIP_LEN);
    station_description[V0_STATION_DESCRIP_LEN]='\0';

    for (i=0; i<num_lines-5; i++)
	if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	    sprintf(err_string, "Error: Text Line# %d of %s was blank.", i+5, 
		    file_name);
	    throw(format_exception(err_string));
	}

    // this should be line 13 (unknown values line)
    if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	throw(format_exception("Error: 13th line (unkown values line) of this V0 entry was empty."));
    }
    cptr = &linebuf[V0_INTEGER_UNKNOWN_COL];
    unknown_int = atoi(cptr);
    cptr = &linebuf[V0_REAL_UNKNOWN_COL];
    unknown_real = atof(cptr);
#ifdef DEBUG
    cout << "unk int="<< unknown_int<< "  unk real="<<unknown_real<<endl;
#endif // DEBUG

    // get the number of integers per line
    if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	throw(format_exception("Error: 14th line (integer count) of this V0 entry was empty."));
    }
    cptr = &linebuf[V0_INTEGER_NUM_LINES_COL];
    num_lines = atoi(cptr);
    cptr = &linebuf[V0_INTEGER_FORMAT_SIZE_COL];
    if ( (cptr2 = strchr(cptr, 'I')) != NULL || 
	 (cptr2 = strchr(cptr, 'i')) != NULL) {
	cptr2++;
	fmt_size=atoi(cptr2);
    } else {
	fmt_size = V0_LINE_SIZE/(V0_NUM_INTS/num_lines);
    }
    num_per_line = (int) ceil((double)V0_NUM_INTS/(double)num_lines);

    // fill the integers array
    for (int_ptr=0,i=0; i< num_lines; i++) {
	if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	    throw(format_exception("Error: Problem reading Integer Array, short read."));
	}
	for (j=0; j< V0_NUM_INTS/num_lines && int_ptr < V0_NUM_INTS; 
	     j++, int_ptr++) {
	    linebuf[fmt_size*(j+1)]='\0';	// terminate the string
	    cptr= &linebuf[j*fmt_size+1];
	    integers[int_ptr] = atoi(cptr);
	}
    }

    // do the same for the reals...
    if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	throw(format_exception("Error:  Line holding real array count of this V0 entry was empty."));
    }
    cptr = &linebuf[V0_REAL_NUM_LINES_COL];
    num_lines = atoi(cptr);
    cptr = &linebuf[V0_REAL_FORMAT_SIZE_COL];
    if ( (cptr2 = strchr(cptr, 'F')) != NULL || 
	 (cptr2 = strchr(cptr, 'f')) != NULL) {
	cptr3 = strchr(cptr2, ')');
	cptr2++;
	fmt_size=atoi(cptr2);
	strcat(real_format, "%");
	if (cptr3 != NULL) {
	    *cptr3 = 'f';
	    cptr3++;
	    *cptr3 = '\0';
	    strcat(real_format, cptr2);
	} else {
	    strncpy(&(real_format[1]), cptr2, 20);
	    real_format[20] = '\0';
	}
    } else {
	fmt_size = V0_LINE_SIZE/(V0_NUM_REALS/num_lines);
    }

    num_per_line = (int) ceil((double)V0_NUM_REALS/(double)num_lines);
    // fill the reals array
    for (int_ptr=0,i=0; i< num_lines; i++) {
	if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	    throw(format_exception("Error: Problem reading Real Array, short read."));
	}
	for (j=0; j< num_per_line && int_ptr < V0_NUM_REALS; j++, int_ptr++) {
	    linebuf[fmt_size*(j+1)]='\0';	// terminate the string
	    cptr= &linebuf[j*fmt_size+1];
	    reals[int_ptr] = atof(cptr);
	}
    }
    // skip past the comments lines
    if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	throw(format_exception("Error:  Line holding comments of this V0 entry was empty."));
    }
    num_lines = atoi(&linebuf[0]);
    // allocate space for the comment lines
    if (num_lines > 0) {
	comment_lines = new char[num_lines * 81 + 1]; // allow for DOS ^M line terminator
	comment_lines[0] = '\0';
    }
    for (i=0; i<num_lines; i++) {
	if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	    throw(format_exception("Error: Problem comment lines; short read."));
	}
	
	if (is_dataless) {
	    // look for the start to end line of this format:
	    // |# 2003/07/18, 00:00:00 to 3000/01/01, 00:00:00 UTC
	    // the XML time format is 2003-07-18T00:00:00.000Z
	    if (strncmp(linebuf, "|#", 2)==0 && 
		strncmp(&linebuf[24], "to", 2)==0)  {
		// we have a dataless start time,  I think anyway
		int s_yr, s_mo, s_dy, s_hr, s_mn, s_sec; 
		int e_yr, e_mo, e_dy, e_hr, e_mn, e_sec, s; 

		char zone[4];
		if ( (s=sscanf(linebuf, "|# %4d/%2d/%2d, %2d:%2d:%2d to %4d/%2d/%2d, %2d:%2d:%2d %3s",
			       &s_yr, &s_mo, &s_dy, &s_hr, &s_mn, &s_sec, 
			       &e_yr, &e_mo, &e_dy, &e_hr, &e_mn, &e_sec, zone)) != 13) {
		    sprintf(err_string, "Error: Dataless timespan for channel %d not formatted correctly: \nLine:%s\n",
			    this->getChannelNumber(), linebuf);
		    throw(format_exception(err_string));
		}
		dataless_start = new char[25];
		dataless_end = new char[25];
		sprintf(dataless_start, "%4d-%02d-%02dT%02d:%02d:%02d.000Z",
			s_yr, s_mo, s_dy, s_hr, s_mn, s_sec);
		sprintf(dataless_end, "%4d-%02d-%02dT%02d:%02d:%02d.000Z",
			e_yr, e_mo, e_dy, e_hr, e_mn, e_sec);
	    }
	}
	strncat(comment_lines, linebuf, 81);
    }
	
    // finally get the number of data points and the format size
    if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	throw(format_exception("Error: Problem reading Data Header, short read."));
    }
    if (! is_dataless) {
	ret = regcomp(&preg, "^ *([0-9]+).*Format *= *.([0-9]+)[A-Z]([0-9]+)", REG_EXTENDED | REG_ICASE);
	if (ret != 0) {
	    regerror(ret, &preg, regex_err, 1024);
	    fprintf(stderr, "Error compiling data header regexp: %s\n", regex_err);
	    return false;
	}
	ret = regexec(&preg, linebuf, 4, pmatch, 0);
	if (ret != 0) {
	    regerror(ret, &preg, regex_err, 1024);
	    fprintf(stderr, "Error executing data header regexp: %s\n", regex_err);
	    regfree(&preg);
	    return false;
	}
	regfree(&preg);
	
	// Number of samples:
	number_samples = atoi(&linebuf[pmatch[1].rm_so]);
	if (number_samples <=0 && !is_dataless) {
	    sprintf(err_string, "Error: Data description line of %s  for channel %d indicated number of samples as %d\n", 
		    file_name, this->getChannelNumber(), number_samples);
	    throw(format_exception(err_string));
	}
	num_data_per_line = atoi(&linebuf[pmatch[2].rm_so]);
	fmt_size = atoi(&linebuf[pmatch[3].rm_so]);

	// allocate enough room for the data
	data = new long[number_samples];
	if (data == NULL) { 
	    fprintf(stderr, "Error allocating memory for data values\n");
	    return false;
	}
	
	num_lines = (int) ceil((double)number_samples/(double)num_data_per_line);
	
	// process the data into the array
	for (int_ptr=0,i=0; i< num_lines; i++) {
	    if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
		throw(format_exception("Error: Problem reading Data Array, short read."));
	    }
	    
	    for (j=0; j< num_data_per_line && int_ptr < number_samples; j++, int_ptr++) {
		linebuf[fmt_size*(j+1)]='\0';	// terminate the string
		cptr= &linebuf[j*fmt_size+1];
		data[int_ptr] = strtol(cptr, NULL, 0);
	    }
	}
    } // end of data reading section
    
    // read the End-of-data line
    if (fgets(linebuf, MAX_LINE_SIZE, fptr) == NULL) {
	sprintf(err_string, "End of data marker missing for channel %d", this->getChannelNumber());
	throw(format_exception(err_string));
    }
    if (strncmp(linebuf, "End-of-data", 11)!=0) {
	sprintf(err_string, "End of data marker malformed for channel %d", this->getChannelNumber());
	throw(format_exception(err_string));
    }
    return true;
}

v0_cosmos::~v0_cosmos() {

    // free up the things allocated
    if (data!=NULL) {
	delete data;
	data = NULL;
    }
    
    if (net_code != NULL) {
	delete net_code;
	net_code = NULL;
    }
    
    if (sta_name != NULL) {
	delete sta_name;
	sta_name = NULL;
    }
    
    if (chan_code != NULL) {
	delete chan_code;
	chan_code = NULL;
    }
    
    if (comment_lines != NULL) {
	delete comment_lines;
	comment_lines = NULL;
    }
    
    if (dataless_start != NULL) {
	delete dataless_start;
	dataless_start = NULL;
    }
    
    if (dataless_end != NULL) {
	delete dataless_end;
	dataless_end = NULL;
    }
}

// note that all index values listed below are -1 from the cosmos document
// integer header value locations 
const int V0_INT_STAGEINDEX = 0;
const int V0_INT_PARAMETERTYPE = 1;
const int V0_INT_CVERSION = 3;
const int V0_INT_STANUMBER = 7;
const int V0_INT_OWNERNETCODE = 11;
const int V0_INT_DLTYPE = 29;
const int V0_INT_DLSERIAL = 31;
const int V0_INT_DLTOTALCHANS = 32;
const int V0_INT_DLRECORDEDCHANS = 33;
const int V0_INT_DLNUMBITS = 34;
const int V0_INT_YEAR = 39;
const int V0_INT_YDAY = 40;
const int V0_INT_MONTH = 41;
const int V0_INT_MDAY = 42;
const int V0_INT_HOUR = 43;
const int V0_INT_MIN = 44;
const int V0_INT_RECORDER_CHAN = 50;	// recorder channel number
const int V0_INT_SENSORTYPE = 51;
const int V0_INT_SENSORSERIAL = 52;
const int V0_INT_SENSORAZI = 53;

// real header value locations 
const int V0_REAL_STALAT = 0;
const int V0_REAL_STALON = 1;
const int V0_REAL_STAELEV = 2;
const int V0_REAL_DLLSB = 21;
const int V0_REAL_DLFULLSCALE = 22;
const int V0_REAL_SECS = 29;
const int V0_REAL_SRATE = 33;
const int V0_REAL_SENSORNATFREQ = 39;
const int V0_REAL_SENSORDAMPING = 40;
const int V0_REAL_SENSORSENSITIVITY = 41;
const int V0_REAL_SENSORFULLSCALEV = 42;
const int V0_REAL_SENSORFULLSCALEG = 43;
const int V0_REAL_SENSORGAIN = 46;
const int V0_REAL_DATALEN = 62;
const int V0_REAL_DATAMAX = 63;
const int V0_REAL_DATAMAXTIME = 64;

int v0_cosmos::getUnknownIntVal() {
    return unknown_int;
}
int v0_cosmos::getStageIndex() {
    return integers[V0_INT_STAGEINDEX];
}
double v0_cosmos::getUnknownRealVal() {
    return unknown_real;
}

// flip the polarity of the data array
void v0_cosmos::switchDataPolarity() {
    int i;

    for (i=0; i < number_samples; i++) {
	data[i] *= -1;
    }
}


///////////////////////////////////////////////////////
// All the get_____() methods are below here....they are
// fairly self explanatory (there is only one setter method).
//
///////////////////////////////////////////////////////

int v0_cosmos::getNumSamples() {
    return number_samples;
}
int v0_cosmos::getCOSMOSversion() {
    return (integers[V0_INT_CVERSION]);
}
int v0_cosmos::getStationNumber() {
    return (integers[V0_INT_STANUMBER]);
}
double v0_cosmos::getStationLatitude() {
    return  (reals[V0_REAL_STALAT]);
}
double v0_cosmos::getStationLongitude() {
    return  (reals[V0_REAL_STALON]);
}
double v0_cosmos::getStationElevation() {
    return  (reals[V0_REAL_STAELEV]);
}
int v0_cosmos::getSampleRate() {
    return (int) (1.0/reals[V0_REAL_SRATE]);
}
char * v0_cosmos::getNetworkCode() {
    // this should really come from integer position 11!
    return net_code;
}
char * v0_cosmos::getOwnerNetworkCode() {
    return _getNetCode(integers[V0_INT_OWNERNETCODE]);
}
char * v0_cosmos::getStationDescription() {
    return station_description;
}

//
// for this ONLY setter method, the length of the station name is not checked
//
void   v0_cosmos::setStationCode(char *sta) {
    if (sta == NULL) return;
    if (sta_name != NULL) {
	delete sta_name;
	sta_name = NULL;
    }
    sta_name = new char[strlen(sta)+1];
    strcpy(sta_name, sta);
}
char * v0_cosmos::getStationCode() {
    if (sta_name == NULL) {
	sta_name = new char[6];
	sprintf(sta_name, "%05d", integers[V0_INTEGER_STA_NUMBER]);
    }
    return sta_name;
}
void v0_cosmos::setChannelCode(char *c) throw( v0_cosmos::format_exception ) {
    chan_code = new char[4];
    strcpy(chan_code, c);
    if (strlen(chan_code) > 3) {
	sprintf(err_string, "Error: channel code %s is too long for the SEED convention", chan_code);
	throw(format_exception(err_string));
    }
}

char * v0_cosmos::getChannelCode() {
    int orient;


    if (chan_code == NULL) {
	chan_code = new char[4];

	orient = integers[V0_INTEGER_ORIENTATION];
	if ( orient > 360 && orient <= 402) {
	    strcpy(chan_code, "HNZ");
	    return chan_code;
	}

	// this is kind of totally arbitrary
	if (orient == 360 || orient == 180) {
	    strcpy(chan_code, "HNN");
	    return chan_code;
	}

	// otherwise it is a East Horizontal component
	if (orient == 90 || orient == 270) {
	    strcpy(chan_code, "HNE");
	    return chan_code;
	}
	if ((orient < 360 && orient > 315) || 
	    (orient > 0 && orient <= 45)   || 
	    (orient > 135 && orient <= 225))  {
	    strcpy(chan_code, "HN2");
	    return chan_code;
	}
	if ( (orient <= 315 && orient > 225)  || 
	     (orient > 45 && orient <= 135))  {
	    strcpy(chan_code, "HN3");
	    return chan_code;
	}
	if (orient == 700) {
	    strcpy(chan_code, "HNU");
	    return chan_code;
	}
	if (orient == 701) {
	    strcpy(chan_code, "HNV");
	    return chan_code;
	}
    }
    return chan_code;
}

int  v0_cosmos::getStartYear() {
    return (integers[V0_INT_YEAR]);
}
int  v0_cosmos::getStartYDay() {
    return (integers[V0_INT_YDAY]);
}
int  v0_cosmos::getStartMonth() {
    return (integers[V0_INT_MONTH]);
}
int  v0_cosmos::getStartMDay() {
    return (integers[V0_INT_MDAY]);
}
int  v0_cosmos::getStartHour() {
    return (integers[V0_INT_HOUR]);
}
int  v0_cosmos::getStartMin() {
    return (integers[V0_INT_MIN]);
}
int  v0_cosmos::getStartSecs() {
    return ((int) reals[V0_REAL_SECS]);
}
int  v0_cosmos::getStartMsecs() {
    int secs,msecs;

    secs = (int) reals[V0_REAL_SECS];
	
    msecs = (int) ((reals[V0_REAL_SECS]-secs)*1000);
    return msecs;
}

long * v0_cosmos::getData() {
    return data;
}
int  v0_cosmos::getDataloggerSerial() {
    return (integers[V0_INT_DLSERIAL]);
}
int  v0_cosmos::getDataloggerNumTotalChannels() {
    return (integers[V0_INT_DLTOTALCHANS]);
}
int  v0_cosmos::getDataloggerNumRecordedChannels() {
    return (integers[V0_INT_DLRECORDEDCHANS]);
}
int  v0_cosmos::getDataloggerDigitizerNumBits() {
    return (integers[V0_INT_DLNUMBITS]);
}

double v0_cosmos::getDataloggerDigitizerLSB() {
    return  (reals[V0_REAL_DLLSB]);
}
double v0_cosmos::getDataloggerDigitizerFullScale() {
    return  (reals[V0_REAL_DLFULLSCALE]);
}
int v0_cosmos::getChannelNumber() {
    return (integers[V0_INT_RECORDER_CHAN]);
}
char * v0_cosmos::getCommentLines() {
    return (comment_lines);
}

char *v0_cosmos::getRealFormatPrintf() {
    return (real_format);
}

// convert the table 9 from code to meta data (human readable) 
// hard coded here to make this go faster since cosmos never
// released meta data tables.
static char dl_type[256];
char * v0_cosmos::getDataloggerType() {
    switch(integers[V0_INT_DLTYPE]) {
	
	// no analogs listed here

    case 100:
	strcpy(dl_type, "DSA-1, Kinemetrics");
	break;
    case 101:
	strcpy(dl_type, "DSA-3, Kinemetrics");
	break;
    case 102:
	strcpy(dl_type, "PDR-1, Kinemetrics");
	break;
    case 103:
	strcpy(dl_type, "PDR-2, Kinemetrics");
	break;
    case 104:
	strcpy(dl_type, "SSA-1, Kinemetrics");
	break;
    case 105:
	strcpy(dl_type, "SSA-2, Kinemetrics");
	break;
    case 106:
	strcpy(dl_type, "SSA-16, Kinemetrics");
	break;
    case 107:
	strcpy(dl_type, "SSR-1, Kinemetrics");
	break;
    case 108:
	strcpy(dl_type, "K2, Kinemetrics");
	break;
    case 109:
	strcpy(dl_type, "Etna, Kinemetrics");
	break;
    case 110:
	strcpy(dl_type, "Mt. Whitney, Kinemetrics");
	break;
    case 111:
	strcpy(dl_type, "Everest, Kinemetrics");
	break;
    case 112:
	strcpy(dl_type, "Makalu, Kinemetrics");
	break;
    case 113:
	strcpy(dl_type, "Etna-2, Kinemetrics");
	break;
    case 120:
	strcpy(dl_type, "QDR, Kinemetrics");
	break;
    case 130:
	strcpy(dl_type, "Granite, Kinemetrics");
	break;

    case 200:
	strcpy(dl_type, "DR-100, Spregnether");
	break;
    case 201:
	strcpy(dl_type, "DR-200, Spregnether");
	break;
    case 202:
	strcpy(dl_type, "DR-300, Spregnether");
	break;
    case 203:
	strcpy(dl_type, "DR-3016, Spregnether");
	break;

    case 300:
	strcpy(dl_type, "DCA-300, Terratech");
	break;
    case 301:
	strcpy(dl_type, "DCA-310, Terratech");
	break;
    case 302:
	strcpy(dl_type, "DCA-333, Terratech");
	break;
    case 310:
	strcpy(dl_type, "IDS-3602, Terratech");
	break;
    case 311:
	strcpy(dl_type, "IDS-3602A, Terratech");
	break;

    case 400:
	strcpy(dl_type, "A700, Geotech");
	break;
    case 401:
	strcpy(dl_type, "A800, Geotech");
	break;
    case 402:
	strcpy(dl_type, "A900, Geotech");
	break;
    case 403:
	strcpy(dl_type, "A900A, Geotech");
	break;

    case 500:
	strcpy(dl_type, "GEOS, US Geological Survey");
	break;
    case 550:
	strcpy(dl_type, "GMS-18, Geosig (NetQuakes)");
	break;

    case 600:
	strcpy(dl_type, "Q4120, Quanterra");
	break;
    case 601:
	strcpy(dl_type, "Q4128a, Quanterra");
	break;
    case 602:
	strcpy(dl_type, "Q730, Quanterra");
	break;
    case 603:
	strcpy(dl_type, "Q736, Quanterra");
	break;
    case 604:
	strcpy(dl_type, "Q980/680, Quanterra");
	break;

    case 700:
	strcpy(dl_type, "72A, Reftek");
	break;
    case 701:
	strcpy(dl_type, "130-01, Reftek");
	break;
    case 702:
	strcpy(dl_type, "130-SM, Reftek");
	break;
    case 703:
	strcpy(dl_type, "130-SMA, Reftek");
	break;
    case 704:
	strcpy(dl_type, "130-ANSS, Reftek");
	break;
    case 705:
	strcpy(dl_type, "130-MC, Reftek");
	break;

    case 1000:
	strcpy(dl_type, "Other");
	break;

    default:
	strcpy(dl_type, "Code not found");
    }
    return dl_type;
}
int  v0_cosmos::getSensorSerial() {
    return (integers[V0_INT_SENSORSERIAL]);
}

// these next 2 dip/azi are derived based on the cosmos table 11
// here they are converted to the SEED convention
int  v0_cosmos::getSensorAzimuth() {
    if (integers[V0_INT_SENSORAZI] > 0 &&
	integers[V0_INT_SENSORAZI] <=360) {
	return integers[V0_INT_SENSORAZI];
    }
    return 0;	// its a vertical component
}
int  v0_cosmos::getSensorDip() {
    if (integers[V0_INT_SENSORAZI] > 0 &&
	integers[V0_INT_SENSORAZI] <= 360) {
	return 0;  // its a horizontal component
    }
    switch (integers[V0_INT_SENSORAZI]) {
    case 400:
    case 402:
	return -90; // up
    case 401:
	return 90; // down
    default:
	return 0;
    }
}
double v0_cosmos::getSensorGain() {
    return  (reals[V0_REAL_SENSORGAIN]);
}
double v0_cosmos::getSensorFullScaleG() {
    return  (reals[V0_REAL_SENSORFULLSCALEG]);
}
double v0_cosmos::getSensorFullScaleV() {
    return  (reals[V0_REAL_SENSORFULLSCALEV]);
}
double v0_cosmos::getSensorNaturalFrequency() {
    return  (reals[V0_REAL_SENSORNATFREQ]);
}
double v0_cosmos::getSensorSensitivity() {
    return  (reals[V0_REAL_SENSORSENSITIVITY]);
}
double v0_cosmos::getSensorDamping() {
    return  (reals[V0_REAL_SENSORDAMPING]);
}
// convert the table 1 from code to meta data (human readable) 
static char phys_param[256];
char * v0_cosmos::getParameterType() {
    switch(integers[V0_INT_PARAMETERTYPE]) {
    case 1:
	strcpy(phys_param, "Acceleration");
	break;
    case 2:
	strcpy(phys_param, "Velocity");
	break;
    case 3:
	strcpy(phys_param, "Displacement (absolute)");
	break;
    case 4:
	strcpy(phys_param, "Displacement (relative)");
	break;
    default:
	strcpy(phys_param, "Code not found");
    }
    return phys_param;
}
// convert the table 10 from code to meta data (human readable) 
// hard coded here to make this go faster since cosmos never
// released meta data tables.
static char sens_type[256];
char * v0_cosmos::getSensorType() {
    switch(integers[V0_INT_SENSORTYPE]) {
	
	// no analogs listed here

    case 2:
	strcpy(sens_type, "FBA-1, Kinemetrics");
	break;
    case 3:
	strcpy(sens_type, "FBA-3, Kinemetrics");
	break;
    case 4:
	strcpy(sens_type, "FBA-11, Kinemetrics");
	break;
    case 5:
	strcpy(sens_type, "FBA-13, Kinemetrics");
	break;
    case 6:
	strcpy(sens_type, "FBA-13DH, Kinemetrics");
	break;
    case 7:
	strcpy(sens_type, "FBA-23, Kinemetrics");
	break;
    case 8:
	strcpy(sens_type, "FBA-23DH, Kinemetrics");
	break;
    case 20:
	strcpy(sens_type, "Episensor, Kinemetrics");
	break;
    case 21:
	strcpy(sens_type, "Episensor ES-U, Kinemetrics");
	break;
    case 22:
	strcpy(sens_type, "Episensor ESDH, Kinemetrics");
	break;
    case 50:
	strcpy(sens_type, "FBX-23, Spregnether");
	break;
    case 51:
	strcpy(sens_type, "FBX-26, Spregnether");
	break;
    case 100:
	strcpy(sens_type, "SSA 120, Terratech");
	break;
    case 101:
	strcpy(sens_type, "SSA 220, Terratech");
	break;
    case 102:
	strcpy(sens_type, "SSA 320, Terratech");
	break;
    case 150:
	strcpy(sens_type, "731A, Wilcoxon");
	break;
    case 200:
	strcpy(sens_type, "CMG-5, Guralp");
	break;
    case 250:
	strcpy(sens_type, "131A-02/01, Reftek");
    case 251:
	strcpy(sens_type, "131A-02/03, Reftek");
    case 252:
	strcpy(sens_type, "131A-02/BH");
    case 253:
	strcpy(sens_type, "131B-01/01");
    case 254:
	strcpy(sens_type, "131B-01/03");
    case 255:
	strcpy(sens_type, "131-8019");
    case 900:
	strcpy(sens_type, "Other, Accelerometer");
    case 9000:
	strcpy(sens_type, "Other");
	break;
    default:
	strcpy(sens_type, "Code not found");
    }
    return sens_type;
}
// convert the table 4 from code to meta data (human readable) 
static char netcode_param[4];
char * _getNetCode(int i) {
    switch(i) {
    case 1: //US Coast and Geodetic Survey
	strcpy(netcode_param, "C_");
	break;
    case 2: // usgs National Strong Motion Program
	strcpy(netcode_param, "NP");
	break;
    case 3:	// bureau of reclamation
	strcpy(netcode_param, "RE");
	break;
    case 4: // U.S. Army Corps of Engineers
	strcpy(netcode_param, "--");
	break;
    case 5: // CGS
	strcpy(netcode_param, "CE");
	break;
    case 6: // CIT
	strcpy(netcode_param, "CI");
	break;
    case 7: // UCB
	strcpy(netcode_param, "BK");
	break;
    case 8: // USGS - Northern Calif Regional Network
	strcpy(netcode_param, "NC");
	break;
    case 9: // UC Santa Barbara
	strcpy(netcode_param, "SB");
	break;
    case 10: // UC San Diego - ANZA
	strcpy(netcode_param, "AZ");
	break;
    case 14: // UNR - W. Great Basin/E. Sierra Nevada
	strcpy(netcode_param, "NN");
	break;
    case 15: // UW - Pacific NW Regional Network
	strcpy(netcode_param, "UW");
	break;
    case 16: // Caltech - Tectonics Observatory
	strcpy(netcode_param, "TO");
	break;
    case 17: // UA - Anchorage Strong Motion Network
	strcpy(netcode_param, "AA");
	break;
    case 20: // Calif. Dept. Water Resources
	strcpy(netcode_param, "WR");
	break;
    case 21: // Pacific Gas & Electric
	strcpy(netcode_param, "PG");
	break;
    case 30: // USGS - National Seismic Network
	strcpy(netcode_param, "US");
	break;
    case 31: // USGS - Golden Network
	strcpy(netcode_param, "GO");
	break;
    case 35: // USGS - Other
	strcpy(netcode_param, "US");
	break;
    case 100: // CWB, Taiwan Central Weather Bureau
	strcpy(netcode_param, "TW");
	break;

    default:
	strcpy(netcode_param, "??");
    }
    return netcode_param;
}
char * v0_cosmos::getDatalessStart()
{
    return dataless_start;
}
char * v0_cosmos::getDatalessEnd()
{
    return dataless_end;
}
bool v0_cosmos::isDataless()
{
    return is_dataless;
}
double v0_cosmos::getDataMaxValue() {
    return  (reals[V0_REAL_DATAMAX]);
}
double v0_cosmos::getDataTimeOfMax() {
    return  (reals[V0_REAL_DATAMAXTIME]);
}
double v0_cosmos::getDataLength() {
    return  (reals[V0_REAL_DATALEN]);
}
