/* snippet.c: Contains the routines for processing a trigger message
 */
/* WARNING: changed some logit entries for UW parsing: PNL, 11/10/97 */

/* Changed 1/13/98 for use with ws_clientII routines: PNL */

/*
Story: we pick up TYPE_TRIG messages which are generated by various detector
algorithms. It's a verbose ascii message which we contacted from exposure to
the Menlo volcano crowd. It lists an event id, event time, and a number of
lines, each listing a station by name, trigger start time, and duration of
data to be saved for that station. Wildcards are permitted in either the
station, network or component field (suggestion made by Steve Malone).
	
We loop over the lines of the message. For each line we construct an array of
implied trace requests. These had better be fewer than the configuration
parameter MaxTrace. We then loop over this array, requesting and disposing of
each trace, on at a time.

As we do so, we might be making premature requests: the waves might not have
reached the sensors yet. The WaveServers are smart enough to tell us if that's
the case, and how long we have to wait. So we wait for the longest wait
period, and then run over the requests again, retrieving any
'stragglers'. After that, we give up.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <earthworm.h>
#include <transport.h>
#include <trace_buf.h>
#include <queue_max_size.h>
#include <ws_clientII.h>
#include "parse_trig.h"
#include "putaway.h"

#include "snippet.h"


extern char	wsIp[MAX_WAVESERVERS][MAX_ADRLEN];
extern char	wsPort[MAX_WAVESERVERS][MAX_ADRLEN];

extern char* TraceBuffer;	/* where we store the trace snippet  */
extern long TraceBufferLen;	/* bytes of largest snippet we're up for - 
			         * from configuration file */
extern TRACE_REQ*   TraceReq; 	/* request forms for wave server. */
extern long wsTimeout;	        /* seconds to wait for reply from ws */
extern long MaxTraces;	        /* max traces per message we'll ever handle  */
extern long TravelTimeout;	/* seconds to wait for wave propogation */
extern int  Urgent;             /* flag to say we're in a hurry, don't dawdle */

/* internal function prototypes */
int snippet2trReq(WS_MENU*, SNIPPET*, TRACE_REQ*, int, int*);
int match(SNIPPET*, WS_PSCN);
int duplicate(WS_PSCN, TRACE_REQ*, int);
int retryServer(TRACE_REQ*, WS_MENU_QUEUE_REC*, int, int);


extern int Debug;

/******************** Message Processing Routine *******************
 *           Take message from argument, create trace              *
 *	snippets, and call the disposal (PutAway) routines         *
 ******************************************************************/
/* story: the trigger message contains lines specifying 'snippets' of trace
   data (EventId, SCN, start time, and duration to save). The S,C, and N
   specification can be a *, meaning wildcard. What we do below is to loop
   over the lines in the message, creating an array of trace request
   structures (TRACE_REQ). We then loop over those, trying to extract the
   trace from the WaveServers.
   
   As we collect, we may be making premature requests. In that case, we're
   told by the WaveServer that we're premature. We accummulate worst case wait
   time for any such stragglers.We wait this amount of time, and then request
   the stragglers again.
   
   */

int SnippetMaker( char *Msg, int* good_count )
{
  int          res;
  int          rc;
  int          i;
  int          ret;
  SNIPPET 	Snppt;   	/* holds params for trace snippet. From 
				   parse_trig.h */
  WS_MENU_QUEUE_REC queue = {NULL,NULL}; /* points to the list of menus; from 
				   ws_clientII.h */
  WS_MENU	menu = NULL; 	/* pointer to WaveServer menu, as built by 
				   ws_client routines; from ws_clientII.h */
  char*	nxtSnippetPtr;	        /* pointer to next line in trigger message */
  double	waitForWaves;	/* wait 'till waveServers have all our stuff */
  int 		iTrReq=0;  	/* running index over TRACE_REQ structures */
  int 		oldNTrRq=0;  	/* temp for storing number of tr req's */
  int 		nTrReq=0;  	/* number of TRACE_REQ structures 
				   from the entire message*/
  int           errors=0;       /* non-fatal error counter */
  int           retried=0;      /* socket retry flag
				 * 0: ok to retry socket
				 * 1: socket has been retried; don't try again.
				 * -1: don't ever retry socket */

  if (Urgent) retried = -1;

  /* Build the current wave server menues
  **************************************/
  for (i=0;i< MAX_WAVESERVERS; i++)
    {
      if ( wsIp[i][0] == 0 ) break;
      ret=wsAppendMenu(wsIp[i], wsPort[i], &queue, wsTimeout);
      switch (ret) {
      case WS_ERR_NONE:
	break;
      case WS_ERR_INPUT:
	logit("e", "SnippetMaker(1): missing input to wsAppendMenu()\n");
	return -1;
	break;
      case WS_ERR_MEMORY:
	logit("e", "SnippetMaker(1): wsAppendMenu failed allocating memory\n");
	wsKillMenu( &queue );
	return -1;
	break;
      case WS_ERR_NO_CONNECTION:
	logit("e","SnippetMaker(1): wsAppendMenu could not get a connection to %s:%s\n",
	      wsIp[i],wsPort[i]);
	errors++;
	break;
      case WS_ERR_SOCKET:
	logit("e", "SnippetMaker(1): wsAppendMenu returned socket error for %s:%s\n",
	      wsIp[i],wsPort[i]);
	errors++;
	break;
      case WS_ERR_TIMEOUT:
	logit("e","SnippetMaker(1): timeout getting menu from %s:%s\n",
	      wsIp[i],wsPort[i]);
	errors++;
	break;
      case WS_ERR_BROKEN_CONNECTION:
	logit("e","SnippetMaker(1): connection to %s:%s broke during menu\n",
	      wsIp[i],wsPort[i]);
	errors++;
	break;
      default:
	logit("e", "SnippetMaker(1): wsAppendMenu returned unknown error %d\n",
	      ret);
	return -1;
      }
      
    }
  if ((menu = queue.head) == NULL ) {
    logit("et","SnippetMaker(1): ERROR No active WaveServers. \n");
    return -1;   /* fatal error */
  }
    
  /* Initialize stuff for loop over snippets
  *****************************************/
  waitForWaves =0;
  nxtSnippetPtr = Msg;  /* set next snippet pointer to start of message */

  /* begin loop over lines in message    
  ***********************************/
  nTrReq = 0; 	/* total number of trace requests from all snippet lines */
  while (rc = parseSnippet(Msg, &Snppt, &nxtSnippetPtr) == RETURN_SUCCESS ) 
    {					/* get next snippet params from msg (level 1)*/
      if (Debug)
	{
	  logit("e","\nreturn from parseSnippet: %d\n",rc);
	  /* WARNING: following changed for UW parsing: PNL, 11/10/97
	     logit("e","eventId=%d, sta=%s, chan=%s, net=%s, startYYMMDD=%s, startHHMMSS=%s, starttime=%lf, duration=%d\n",
	     Snppt.eventId,Snppt.sta,Snppt.chan,Snppt.net,Snppt.startYYMMDD,Snppt.startHHMMSS,Snppt.starttime,Snppt.duration	); */
	  logit("e","eventId=%d, sta=%s, chan=%s, net=%s, startstr=%s, starttime=%lf, duration=%d\n",
		Snppt.eventId,Snppt.sta,Snppt.chan,Snppt.net,Snppt.startstr,Snppt.starttime,Snppt.duration	);

	}

      /* create requests implied by this snippet - ( it might include wildcards) 
      **************************************************************************/
      /* routine below will create the request structures. It has access to
	 the WaveServer Menu lists.  A little clumsiness: snippet2trReq only
	 fills in the SCN name portion of the trace request.  The rest is done
	 in the loop after the call.*/
      
      oldNTrRq=nTrReq; /* save current value of trace reqeusts */
      rc = snippet2trReq( &menu, &Snppt, TraceReq, MaxTraces, &nTrReq ); 
      if (rc==WS_ERR_BUFFER_OVERFLOW)  
	/* then we've run out of request structures. */
	{  
	  errors ++;
	}
      if (Debug)  
	logit("e","return from snippet2trReq: total of %d requests so far\n",
	      nTrReq);
      if (oldNTrRq == nTrReq)  /* then this snippet generated no requests. */
	logit("e"," SnippetMaker(1): WARNING: %s %s %s in event %d either duplicated or not found \n",
	      Snppt.sta,Snppt.chan,Snppt.net,Snppt.eventId);
      for(iTrReq=oldNTrRq; iTrReq<nTrReq; iTrReq++)	/* start of loop over requests (level 2) */
	{		
	  TraceReq[iTrReq].reqStarttime = Snppt.starttime;
	  TraceReq[iTrReq].reqEndtime = Snppt.starttime + Snppt.duration ;
	  TraceReq[iTrReq].pBuf = TraceBuffer;
	  TraceReq[iTrReq].bufLen = TraceBufferLen;
	  TraceReq[iTrReq].timeout = wsTimeout;
  /* Following commented out since it was never implemented in the server */
	  /*  TraceReq[iTrReq].partial = 0; */	/* start proud: 
                                                 * we don't want partial data */

	  if (Debug)  /* dump the newly created request */
	    {
	      logit("e","Request %d:\n sta=%s, chan=%s, net=%s, reqStarttime=%lf, reqEndtime=%lf, timeout=%ld\n",
		    iTrReq,TraceReq[iTrReq].sta,TraceReq[iTrReq].chan,TraceReq[iTrReq].net,
		    TraceReq[iTrReq].reqStarttime,TraceReq[iTrReq].reqEndtime,TraceReq[iTrReq].timeout);
	    }
	}			/* end of loop over requests    (level 2) */
      if (Debug)  logit("e","\n");

    }                      /* end of loop over lines in message    (level 1) */

  /* Call the Put Away initializer
  ********************************/
  /* This routine is responsible for initializing the scheme for disposing
     of the trace snippets we're hopefully about to produce */
  if(Debug)logit("e","calling PA_next_ev\n");
  rc = PA_next_ev( TraceReq, nTrReq, Debug );
  if (rc == RETURN_FAILURE) return -1;   /* fatal error */

  /* Debug: dump out the accummulated trace requests */
  if(Debug)
    {
      logit("e","Trace requests from message %d:\n",Snppt.eventId);
      for ( iTrReq=0; iTrReq<nTrReq; iTrReq++)
	logit("e"," %s %s %s from: %lf to: %lf\n",
	      TraceReq[iTrReq].sta,TraceReq[iTrReq].chan,TraceReq[iTrReq].net,
	      TraceReq[iTrReq].reqStarttime,TraceReq[iTrReq].reqEndtime);
    }

  /* begin loop over retrieving and disposing of the trace snippets  (level 1)
  **************************************************************************/
  for ( iTrReq=0; iTrReq<nTrReq; iTrReq++)
    {
      /* get this trace; rummage through the servers we've been told about */
      rc = wsGetTraceBin( &(TraceReq[iTrReq]), &queue, wsTimeout ); 

      switch (rc) {
      case WS_ERR_NONE:
	if (Debug) logit("et",
           "SnippetMaker(1): trace %s %s %s: went ok first time. Got %ld bytes\n",
			 TraceReq[iTrReq].sta, TraceReq[iTrReq].chan, 
			 TraceReq[iTrReq].net, TraceReq[iTrReq].actLen); 
	if (retried > 0) retried = 0;
	break;

      case WS_WRN_FLAGGED:
	logit("e", "server has no data for <%s:%s:%s>\n", 
	      TraceReq[iTrReq].sta, TraceReq[iTrReq].chan,
	      TraceReq[iTrReq].net);
	continue;

      case WS_ERR_EMPTY_MENU:
	/* something is seriously wrong; give up */
	logit("e", "SnippetMaker(1): wsGetTraceBin reports no menues\n");
	return RETURN_FAILURE;

      case  WS_ERR_SCN_NOT_IN_MENU:
	/* This shouldn't happen, but we'll blunder on */
	logit("e", "SnippetMaker(1): wsGetTraceBin reports SCN not in menu: <%s:%s:%s>\n",

	      TraceReq[iTrReq].sta, TraceReq[iTrReq].chan, TraceReq[iTrReq].net);
	errors++;
	continue;

      case WS_ERR_NO_CONNECTION:
	/* socket to server was marked as dead, so skip it and move on */
	errors++;
	continue;
      
      case WS_ERR_BUFFER_OVERFLOW:
	/* Hard to predict why this happened; we'll try to continue */
	logit("et"," SnippetMaker(1): trace %s.%s.%s overflowed buffer\n",
	      TraceReq[iTrReq].sta, TraceReq[iTrReq].chan, TraceReq[iTrReq].net); 
	errors ++;
	/* socket has been closed, so we need to open it */
	/* consider trying to get trace again */
	ret = retryServer( &(TraceReq[iTrReq]), &queue, wsTimeout, 0);
	/* Ignore the error condition; the server is marked as dead,
	 * so further attempts to it will be skipped. */
	continue;

	/* For the following error conditions, the connection has failed.
	 * We'll try once more to make a connection and get a trace.
	 * If it works, we'll mark to socket as good on the assumption it
	 * was a one-time flaw.
	 * If it fails, the socket is marked as bad (menu-sock = -1), so
	 * it will be skipped on later TraceReqs. 
	 */
      case WS_ERR_BROKEN_CONNECTION:
	errors++;
	logit("e", "SnippetMaker(1): broken connection while getting %s.%s.%s\n",
	      TraceReq[iTrReq].sta, TraceReq[iTrReq].chan, 
	      TraceReq[iTrReq].net);
	if (retried) continue;
	retried = 1;
	if (retryServer( &(TraceReq[iTrReq]), &queue, wsTimeout, 1) !=
	    WS_ERR_NONE)
	  continue;
	if (retried > 0) retried = 0;
	break;
      case WS_ERR_TIMEOUT:
	errors++;
	logit("e", "SnippetMaker(1): timeout while getting %s.%s.%s\n",
	      TraceReq[iTrReq].sta, TraceReq[iTrReq].chan, 
	      TraceReq[iTrReq].net);
	if (retried) continue;
	retried = 1;
	/* Do we really want to try getting trace data again?
	 * If the socket is really going bad, it may time out while
	 * trying to connect. */
	if (retryServer( &(TraceReq[iTrReq]), &queue, wsTimeout, 1) !=
	    WS_ERR_NONE)
	  continue;
	if (retried > 0) retried = 0;
	break;
      default:
	errors++;
	logit("e", "SnippetMaker(1): wsGetTraceBin returned unknown error %s\n",
	      ret);
	continue;
      }
      
      if (Debug)
	{
	  logit("e","return from wsGetTraceBin: %d\n",rc);
	  logit("e","actStarttime=%lf, actEndtime=%lf, actLen=%ld\n",
		TraceReq[iTrReq].actStarttime,TraceReq[iTrReq].actEndtime,
		TraceReq[iTrReq].actLen);	
	  {
	    TRACE_HEADER* pth;
	    pth=(TRACE_HEADER*)TraceReq[iTrReq].pBuf;
	    logit("e","Values of first TRACE_HEADER message:\n");
	    logit("e","TRACE_HEADER->pinno: %d \n",pth->pinno);
	    logit("e","TRACE_HEADER->nsamp: %d \n",pth->nsamp);
	    logit("e","TRACE_HEADER->starttime: %lf \n",pth->starttime);
	    logit("e","TRACE_HEADER->endtime: %lf \n",pth->endtime);
	    logit("e","TRACE_HEADER->samprate: %lf \n",pth->samprate);
	    logit("e","TRACE_HEADER->sta: %s \n",pth->sta);
	    logit("e","TRACE_HEADER->net: %s \n",pth->net);
	    logit("e","TRACE_HEADER->chan: %s \n",pth->chan);
	    logit("e","TRACE_HEADER->datatype: %s \n",pth->datatype);
	  }
	}

      /* accummulate wait times
      ************************/
      /* We collect wait times, wait the max of those, and then issue requests
	 for those. It's assumed that fancier
	 strategies will be developed. */
      if( TraceReq[iTrReq].waitSec > 0 )
	{
	  if (Debug) logit("e","trace %s %s %s has wait time: %lf\n",
			      TraceReq[iTrReq].sta, TraceReq[iTrReq].chan, 
			      TraceReq[iTrReq].net,TraceReq[iTrReq].waitSec);
	  if( waitForWaves< TraceReq[iTrReq].waitSec) 
	    waitForWaves = TraceReq[iTrReq].waitSec;
	}

      if (TraceReq[iTrReq].actLen == 0) 
	{
	  logit("e", "SnippetMaker(1): unflagged zero length trace from %s.%s.%s\n",
		TraceReq[iTrReq].sta, TraceReq[iTrReq].chan, 
		TraceReq[iTrReq].net);
	  errors++;
	  continue;
	}
      
      /* Now call the Put Away snippet disposer
      ****************************************/
      if(Debug)logit("e","calling PA_next\n");
      rc = PA_next( &(TraceReq[iTrReq]), Snppt.eventId );
      if (rc == RETURN_FAILURE) {
	errors ++;  /* non-fatal error */
      } else {
	(*good_count)++;
      }
	
    }  
  /* end of loop getting and disposing snippets: the first attempt to get
     trace (level 1) */

  if (Urgent) goto Done;
  
  /* Wait for stragglers if there are any
  ***************************************/
  /* We now know how long till the last straggler arrives in the wave
     server. Wait that long, and make a pass over the request structure array
     again, collecting all stragglers. */
  if ( waitForWaves > TravelTimeout ) 
    waitForWaves=TravelTimeout;  /* don't wait longer than limit */
  if (waitForWaves <= 0) goto Done;
  if(Debug)logit("e","Waiting for stragglers: %lf seconds\n", waitForWaves);
  sleep_ew( (long)(waitForWaves*1000) );

  /* Rebuild the menu
  ******************/
  wsKillMenu( &queue );
  for (i=0;i< MAX_WAVESERVERS; i++)
    {
      if ( wsIp[i][0] == 0 ) break;
      ret = wsAppendMenu(wsIp[i], wsPort[i], &queue, wsTimeout);
      switch (ret) {
      case WS_ERR_NONE:
	break;
      case WS_ERR_INPUT:
	logit("e", "SnippetMaker(2): missing input to wsAppendMenu()\n");
	return -1;
	break;
      case WS_ERR_MEMORY:
	logit("e", "SnippetMaker(2): wsAppendMenu failed allocating memory\n");
	wsKillMenu( &queue );
	return -1;
	break;
      case WS_ERR_NO_CONNECTION:
	logit("e","SnippetMaker(2): wsAppendMenu could not get a connection to %s:%s\n",
	      wsIp[i],wsPort[i]);
	errors++;
	break;
      case WS_ERR_SOCKET:
	logit("e", "SnippetMaker(2): wsAppendMenu returned socket error for %s:%s\n",
	      wsIp[i],wsPort[i]);
	errors++;
	break;
      case WS_ERR_TIMEOUT:
	logit("e","SnippetMaker(2): timeout getting menu from %s:%s\n",
	      wsIp[i],wsPort[i]);
	errors++;
	break;
      case WS_ERR_BROKEN_CONNECTION:
	logit("e","SnippetMaker(2): connection to %s:%s broke during menu\n",
	      wsIp[i],wsPort[i]);
	errors++;
      default:
	logit("e", "SnippetMaker(2): wsAppendMenu returned unknown error %d\n",
	      ret);
	return -1;
      }
    }

  /* loop over trace requests again, collecting stragglers 
  *******************************************************/
  for (iTrReq=0; iTrReq<nTrReq; iTrReq++)
    {			      /* begin loop over stragglers    (level 1) */
      if( TraceReq[iTrReq].waitSec > 0 )  /* then this one is a straggler */
	{
	  if(Debug)logit("e","Asking for straggler %s %s %s\n", 
			 TraceReq[iTrReq].sta,TraceReq[iTrReq].chan,
			 TraceReq[iTrReq].net);
  /* Following commented out since it was never implemented in the server */
	  /* TraceReq[iTrReq].partial =1; */  /* We're desperate now: we'll accept partial data */
	  
	  /**** Get it from wave server ****/
	  rc = wsGetTraceBin( &(TraceReq[iTrReq]), &queue, wsTimeout ); 
	  switch (rc) {
	  case WS_ERR_NONE:
	    if (Debug) logit("et",
	     "SnippetMaker(2): trace %s %s %s: went ok first time. Got %ld bytes\n",
			     TraceReq[iTrReq].sta, TraceReq[iTrReq].chan, 
			     TraceReq[iTrReq].net, TraceReq[iTrReq].actLen); 
	    if (retried > 0) retried = 0;
	    break;
	    
	  case WS_WRN_FLAGGED:
	    logit("e", "server %s:%s has no data for <%s:%s:%s>\n", 
		  TraceReq[iTrReq].sta, TraceReq[iTrReq].chan,
		  TraceReq[iTrReq].net);
	    continue;
	    
	  case WS_ERR_EMPTY_MENU:
	    /* something is seriously wrong; give up */
	    logit("e", "SnippetMaker(2): wsGetTraceBin reports no menues\n");
	    return RETURN_FAILURE;
	    
	  case  WS_ERR_SCN_NOT_IN_MENU:
	    /* This shouldn't happen, but we'll blunder on */
	    logit("e", "SnippetMaker(2): wsGetTraceBin reports SCN not in menu: <%s:%s:%s>\n",
		  
		  TraceReq[iTrReq].sta, TraceReq[iTrReq].chan, TraceReq[iTrReq].net);
	    errors++;
	    continue;
	    
	  case WS_ERR_NO_CONNECTION:
	    /* socket to server was marked as dead, so skip it and move on */
	    errors++;
	    continue;
	    
	  case WS_ERR_BUFFER_OVERFLOW:
	    /* Hard to predict why this happened; we'll try to continue */
	    logit("et"," SnippetMaker(2): trace %s.%s.%s overflowed buffer\n",
		  TraceReq[iTrReq].sta, TraceReq[iTrReq].chan, TraceReq[iTrReq].net); 
	    errors ++;
	    /* socket has been closed, so we need to open it */
	    /* consider trying to get trace again */
	    ret = retryServer( &(TraceReq[iTrReq]), &queue, wsTimeout, 0);
	    /* Ignore the error condition; the server is marked as dead,
	     * so further attempts to it will be skipped. */
	    continue;
	    
	    /* For the following error conditions, the connection has failed.
	     * We'll try once more to make a connection and get a trace.
	     * If it works, we'll mark to socket as good on the assumption it
	     * was a one-time flaw.
	     * If it fails, the socket is marked as bad (menu-sock = -1), so
	     * it will be skipped on later TraceReqs. 
	     */
	  case WS_ERR_BROKEN_CONNECTION:
	    errors++;
	    logit("e", "SnippetMaker(2): broken connection while getting %s.%s.%s\n",
		  TraceReq[iTrReq].sta, TraceReq[iTrReq].chan, 
		  TraceReq[iTrReq].net);
	    if (retried) continue;
	    retried = 1;
	    if (retryServer( &(TraceReq[iTrReq]), &queue, wsTimeout, 1) !=
		WS_ERR_NONE)
	      continue;
	    if (retried > 0) retried = 0;
	    break;
	  case WS_ERR_TIMEOUT:
	    errors++;
	    logit("e", "SnippetMaker(2): timeout while getting %s.%s.%s\n",
		  TraceReq[iTrReq].sta, TraceReq[iTrReq].chan, 
		  TraceReq[iTrReq].net);
	    if (retried) continue;
	    retried = 1;
	    /* Do we really want to try getting trace data again?
	     * If the socket is really going bad, it may time out while
	     * trying to connect. */
	    if (retryServer( &(TraceReq[iTrReq]), &queue, wsTimeout, 1) !=
		WS_ERR_NONE)
	      continue;
	    if (retried > 0) retried = 0;
	    break;
	  default:
	    errors++;
	    logit("e", "SnippetMaker(2): wsGetTraceBin returned unknown error %s\n",
		  ret);
	    continue;
	  }
	  
	  
	  /* Now call the Put Away snippet disposer
	  ****************************************/
	  if(Debug)logit("e","calling PA_next\n");
	  rc = PA_next( &(TraceReq[iTrReq]), Snppt.eventId );
	  if (rc == RETURN_FAILURE) {
	    errors ++;  /* non-fatal error */
	  } else {
	    (*good_count)++;
	  }

	}
    }		/* end of loop collecting stragglers  (level 1) */

Done:
  wsKillMenu( &queue );
  if(Debug)logit("e","calling PA_end_ev\n");
  PA_end_ev();		
  
  return ( errors );
}


/******************************************************************************
 * given a pointer to a snippet structure (as parsed from the trigger message)*
 * and the array of blank trace request structures, fill them with the        *
 * requests implied by the snippet.                                           *
 * Don't overflow, and return the number of requests generated                *
 * args: pmenu: pointer to the client routine's menu list                     *
 *	pSnppt:	pointer to the structure containing the parsed trigger line   *
 *	     from the trigger message.                                        *
 *	pTrReq: pointer to arrray of trace request forms		      *
 *	maxTraceReq: size of this array					      *
 *	pnTraceReq: pointer to count of currently used trace request forms    *
 ******************************************************************************/
int snippet2trReq(WS_MENU* pmenu, SNIPPET* pSnppt, TRACE_REQ* pTrReq, 
		  int maxTraceReq, int* pnTraceReq )
{
  int ret =  WS_ERR_SCN_NOT_IN_MENU;
  WS_MENU menu = *pmenu;

  int iTrReq=0; /* which trace request form to fill out next */

  while ( menu )
    {
      WS_PSCN pscn = menu->pscn;
      while ( pscn )
	{
	  if ( match(pSnppt, pscn)==1 && duplicate(pscn, pTrReq, *pnTraceReq)==0)
	    {
	      ret = WS_ERR_NONE;
	      strcpy( pTrReq[*pnTraceReq].sta, pscn->sta  );
	      strcpy( pTrReq[*pnTraceReq].net, pscn->net  );
	      strcpy( pTrReq[*pnTraceReq].chan,pscn->chan );
	      (*pnTraceReq)++;
	      if( *pnTraceReq >= maxTraceReq) goto overflow;
	    }
	  pscn = pscn->next;
	}
      menu = menu->next;
    }
  return ret;

overflow: 
  logit("e","snippet2trReq: overflowed trace request array\n");
  return WS_ERR_BUFFER_OVERFLOW;
}
/******************************************************************************
 * helper routine for snippet2trReq above: See if the SCN name in the Snippet *
 * structure matches the menu item. Wildcards allowed                         *
 ******************************************************************************/
int match( SNIPPET* pSnppt, WS_PSCN pscn)
{
  int staWild =0;
  int netWild =0;
  int chanWild=0;
	
  int staMatch =0;
  int netMatch =0;
  int chanMatch=0;
	
  if (strcmp( pSnppt->sta , "*") == 0)  staWild =1;
  if (strcmp( pSnppt->chan, "*") == 0)  chanWild=1;
  if (strcmp( pSnppt->net , "*") == 0)  netWild =1;

  if ( strcmp( pSnppt->sta, pscn->sta )==0 ) staMatch=1;
  if ( strcmp( pSnppt->net, pscn->net )==0 ) netMatch=1;
  if ( strcmp( pSnppt->chan, pscn->chan )==0 ) chanMatch=1;

  if ( staWild  ==1 ||  staMatch==1)  staMatch =1;
  if ( netWild  ==1 ||  netMatch==1)  netMatch =1;
  if ( chanWild ==1 || chanMatch==1) chanMatch =1;

  if (staMatch+netMatch+chanMatch == 3) 
    return(1);
  else
    return(0);
}
/******************************************************************************
 * helper routine for snippet2trReq above: See if the SCN name in the Snippet *
 * structure is a duplicate of any existing request                           *
 ******************************************************************************/
int duplicate( WS_PSCN pscn, TRACE_REQ* pTrReq, int nTraceReq)
{
  int i;
  /* logit("e","entered duplicate. nTraceReq=%d\n",nTraceReq); */ /* DEBUG */
  for (i=0; i<nTraceReq; i++)
    {
      /* logit("e","duplicate: comparing %s %s %s with %s %s %s\n",
	    pTrReq[i].sta, pTrReq[i].net, pTrReq[i].chan,
	    pscn->sta,     pscn->net,     pscn->chan); */
      if( strcmp( pTrReq[i].sta, pscn->sta  )==0 &&
	  strcmp( pTrReq[i].net, pscn->net  )==0  &&
	  strcmp( pTrReq[i].chan,pscn->chan )==0   ) 
	return(1); /* meaning: yes, it's a duplicate */
    }
  return(0); /* 'twas not a duplicate */
}

/************************************************************************
 * retryServer: try to reconnect to a server; timeout is in millisconds *
 *  getTrace flag: =0 means don't get the trace, just connect to server *
 ************************************************************************/
int retryServer( TRACE_REQ* getThis, WS_MENU_QUEUE_REC* menu_queue,
		 int timeout, int getTrace )
{
  int ret;
  WS_MENU menu = NULL;
  WS_PSCN pscn = NULL;
  
  ret = wsSearchSCN( getThis, &menu, &pscn, menu_queue);
  if (ret != WS_ERR_NONE) goto Abort;
    
  if (menu->sock > 0) wsDetachServer( menu );
  
  ret = wsAttachServer( menu, timeout );
  if (ret != WS_ERR_NONE) goto Abort;

  if (getTrace) 
    ret = wsGetTraceBin( getThis, menu_queue, timeout);
  
  return ret;

Abort:
  logit("e", "retryServer: error %d for server %s:%s\n", ret, menu->addr, 
	menu->port);
  return ret;
}
